/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.async;

import com.sleepycat.je.utilint.LongAvgRate;
import java.nio.ByteBuffer;
import java.util.Deque;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import oracle.kv.impl.async.IOBufSliceImpl;

public class IOBufferPool {
    private static volatile Logger logger;
    private static final String WARN_LEAK = "oracle.kv.async.bufpool.warn.leak";
    private static final boolean warnLeak;
    private static final Timer timer;
    private static final long detectionPeriodMillis = 1000L;
    private static final int numPeriodsPerLogging = 60;
    private static final String BUFPOOL_DISABLED = "oracle.kv.async.bufpool.disabled";
    private static final boolean bufPoolDisabled;
    private static final String CHANNEL_INPUT_BUF_SIZE = "oracle.kv.async.bufpool.channelinput.bufsize";
    private static final int CHANNEL_INPUT_BUF_SIZE_DEFAULT = 4096;
    private static final int channelInputBufSize;
    private static final String MESSAGE_OUTPUT_BUF_SIZE = "oracle.kv.async.bufpool.messageoutput.bufsize";
    private static final int MESSAGE_OUTPUT_BUF_SIZE_DEFAULT = 128;
    private static final int messageOutputBufSize;
    private static final String CHANNEL_OUTPUT_BUF_SIZE = "oracle.kv.async.bufpool.channeloutput.bufsize";
    private static final int CHANNEL_OUTPUT_BUF_SIZE_DEFAULT = 64;
    private static final int channelOutputBufSize;
    private static final String IO_BUFPOOL_SIZE_HEAP_PERCENT = "oracle.kv.async.bufpool.size.heap.percentage";
    private static final int IO_BUFPOOL_SIZE_HEAP_PERCENT_DEFAULT = 1;
    private static final long IO_BUFPOOL_SIZE_MAX_DEFAULT = 0x40000000L;
    private static final int poolSizeHeapPercent;
    private static final long maxPoolBytes;
    public static final IOBufferPool CHNL_IN_POOL;
    public static final IOBufferPool MESG_OUT_POOL;
    public static final IOBufferPool CHNL_OUT_POOL;
    private final String name;
    protected final int bufsize;
    private final Deque<ByteBuffer> freeBufs = new ConcurrentLinkedDeque<ByteBuffer>();
    private final long maxPoolSize;
    private final AtomicInteger currPoolSize = new AtomicInteger(0);
    private final AtomicInteger inUse = new AtomicInteger(0);
    private final AtomicInteger minInUse = new AtomicInteger(0);

    public static void setLogger(Logger l) {
        if (logger != null) {
            throw new IllegalStateException("Logger already set");
        }
        logger = l;
        if (warnLeak) {
            logger.log(Level.INFO, "Warning enabled for IO buffer pool leak");
        }
    }

    protected IOBufferPool(String name, int bufsize) {
        this.name = name;
        this.bufsize = bufsize;
        this.maxPoolSize = maxPoolBytes / (long)bufsize;
        timer.schedule((TimerTask)new LeakDetectionTask(), 0L, 1000L);
    }

    ByteBuffer allocPooled() {
        if (bufPoolDisabled) {
            return this.allocDiscarded();
        }
        ByteBuffer buf = this.allocate();
        if (buf != null) {
            this.inUse.incrementAndGet();
        }
        return buf;
    }

    private ByteBuffer allocate() {
        int size;
        do {
            ByteBuffer buf;
            if ((buf = this.freeBufs.poll()) != null) {
                return buf;
            }
            size = this.currPoolSize.get();
            if ((long)size < this.maxPoolSize) continue;
            return null;
        } while (!this.currPoolSize.compareAndSet(size, size + 1));
        return ByteBuffer.allocate(this.bufsize);
    }

    ByteBuffer allocDiscarded() {
        return ByteBuffer.allocate(this.bufsize);
    }

    void deallocate(ByteBuffer buffer) {
        if (bufPoolDisabled) {
            return;
        }
        buffer.clear();
        this.freeBufs.push(buffer);
        int val = this.inUse.decrementAndGet();
        this.updateMinInUse(val);
    }

    private void updateMinInUse(int val) {
        this.minInUse.updateAndGet(curr -> Math.min(curr, val));
    }

    public int getMinNumInUse() {
        int min = this.minInUse.get();
        int use = this.inUse.get();
        this.updateMinInUse(use);
        return min;
    }

    public int getNumInUse() {
        return this.inUse.get();
    }

    public void clearUse() {
        this.minInUse.set(0);
        this.inUse.set(0);
        this.freeBufs.clear();
    }

    static {
        warnLeak = Boolean.getBoolean(WARN_LEAK);
        timer = new Timer("IOBufferPool leak detection", true);
        bufPoolDisabled = Boolean.getBoolean(BUFPOOL_DISABLED);
        channelInputBufSize = Integer.getInteger(CHANNEL_INPUT_BUF_SIZE, 4096);
        messageOutputBufSize = Integer.getInteger(MESSAGE_OUTPUT_BUF_SIZE, 128);
        channelOutputBufSize = Integer.getInteger(CHANNEL_OUTPUT_BUF_SIZE, 64);
        poolSizeHeapPercent = Integer.getInteger(IO_BUFPOOL_SIZE_HEAP_PERCENT, 1);
        maxPoolBytes = Math.min(Runtime.getRuntime().maxMemory() * (long)poolSizeHeapPercent / 100L, 0x40000000L);
        CHNL_IN_POOL = new IOBufferPool("Channel input", channelInputBufSize);
        MESG_OUT_POOL = new IOBufferPool("Message output", messageOutputBufSize);
        CHNL_OUT_POOL = new IOBufferPool("Channel output", channelOutputBufSize);
    }

    private class LeakDetectionTask
    extends TimerTask {
        private long numPeriodsLeaking;
        private final LongAvgRate leakingRate = new LongAvgRate("leaking rate", 1000L, TimeUnit.SECONDS);

        private LeakDetectionTask() {
        }

        @Override
        public void run() {
            int min = IOBufferPool.this.getMinNumInUse();
            if (min == 0) {
                this.numPeriodsLeaking = 0L;
                return;
            }
            ++this.numPeriodsLeaking;
            this.leakingRate.add(min, System.currentTimeMillis());
            if (this.numPeriodsLeaking % 60L == 0L && logger != null && warnLeak) {
                logger.log(Level.INFO, () -> String.format("Possible buffer pool [%s] leaking for the last %s seconds, min number of buffers in use during the last %s second: %s, leaking rate: %sremaining slices: %s", IOBufferPool.this.name, this.numPeriodsLeaking * 1000L / 1000L, 1L, min, this.leakingRate.get(), IOBufSliceImpl.remainingSlicesToString()));
            }
        }
    }
}

