/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.security.filestore;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import oracle.kv.impl.security.PasswordStore;
import oracle.kv.impl.security.PasswordStoreException;
import oracle.kv.impl.security.util.SecurityUtils;

public class FileStore
implements PasswordStore {
    private final File storeLocation;
    private SecretHash secretHash = null;
    private boolean modified = false;
    private static final String LOGIN_USER_PREFIX = "login.user.";
    private static final String LOGIN_PW_PREFIX = "login.password.";
    private static final String SECRET_PREFIX = "secret.";

    FileStore(File storeFile) {
        this.storeLocation = storeFile;
    }

    @Override
    public boolean create(char[] passphrase) throws IOException {
        this.assertNotInitialized();
        if (passphrase != null) {
            throw new UnsupportedOperationException("Passphrases are not supported");
        }
        if (this.storeLocation.exists()) {
            throw new PasswordStoreException("A file already exists at this location");
        }
        File parentDir = this.storeLocation.getParentFile();
        if (parentDir == null) {
            parentDir = new File(".");
        }
        if (!parentDir.exists() || !parentDir.isDirectory()) {
            throw new PasswordStoreException("The directory for the password file does not exist");
        }
        if (!parentDir.canWrite()) {
            throw new PasswordStoreException("The directory for the password file is not writable");
        }
        SecretHash newSecretHash = new SecretHash();
        newSecretHash.write(this.storeLocation);
        if (!SecurityUtils.makeOwnerAccessOnly((File)this.storeLocation)) {
            throw new PasswordStoreException("Unable to set access permissions for file. Correct manually before using the password store");
        }
        try {
            newSecretHash.read(this.storeLocation);
        }
        catch (IOException ioe) {
            throw new PasswordStoreException("Error retrieving passwords from file", ioe);
        }
        this.secretHash = newSecretHash;
        return true;
    }

    @Override
    public boolean open(char[] passphrase) throws IOException {
        this.assertNotInitialized();
        if (!this.storeLocation.exists()) {
            throw new PasswordStoreException("No file exists at this location");
        }
        if (passphrase != null) {
            throw new UnsupportedOperationException("Passphrases are not supported by this implementation");
        }
        SecretHash newSecretHash = new SecretHash();
        newSecretHash.read(this.storeLocation);
        this.secretHash = newSecretHash;
        return true;
    }

    @Override
    public Collection<String> getSecretAliases() throws IOException {
        this.assertInitialized();
        HashSet<String> secretAliases = new HashSet<String>();
        Iterator e = this.secretHash.aliases();
        while (e.hasNext()) {
            String alias = (String)e.next();
            if (!alias.startsWith(SECRET_PREFIX)) continue;
            alias = alias.substring(SECRET_PREFIX.length());
            secretAliases.add(alias);
        }
        return secretAliases;
    }

    @Override
    public char[] getSecret(String alias) throws IOException {
        this.assertInitialized();
        return this.secretHash.getSecret(SECRET_PREFIX + alias);
    }

    @Override
    public boolean setSecret(String alias, char[] secret) throws IOException {
        this.assertInitialized();
        String internalAlias = SECRET_PREFIX + alias;
        boolean updated = this.secretHash.containsAlias(internalAlias);
        this.secretHash.setSecret(internalAlias, secret);
        this.modified = true;
        return updated;
    }

    @Override
    public boolean deleteSecret(String alias) throws IOException {
        this.assertInitialized();
        String internalAlias = SECRET_PREFIX + alias;
        if (this.secretHash.containsAlias(internalAlias)) {
            this.secretHash.deleteSecret(internalAlias);
            this.modified = true;
            return true;
        }
        return false;
    }

    @Override
    public Collection<PasswordStore.LoginId> getLogins() throws IOException {
        this.assertInitialized();
        HashSet<PasswordStore.LoginId> logins = new HashSet<PasswordStore.LoginId>();
        Iterator e = this.secretHash.aliases();
        while (e.hasNext()) {
            String alias = (String)e.next();
            if (!alias.startsWith(LOGIN_USER_PREFIX)) continue;
            String db = alias.substring(LOGIN_USER_PREFIX.length());
            String user = new String(this.secretHash.getSecret(alias));
            logins.add(new PasswordStore.LoginId(db, user));
        }
        return logins;
    }

    @Override
    public boolean setLogin(PasswordStore.LoginId login, char[] password) throws IOException {
        this.assertInitialized();
        String dbAlias = LOGIN_USER_PREFIX + login.getDatabase();
        boolean exists = this.secretHash.containsAlias(dbAlias);
        this.secretHash.setSecret(dbAlias, login.getUser().toCharArray());
        this.secretHash.setSecret(LOGIN_PW_PREFIX + login.getDatabase(), password);
        this.modified = true;
        return exists;
    }

    @Override
    public PasswordStore.LoginId getLoginId(String database) throws IOException {
        this.assertInitialized();
        char[] user = this.secretHash.getSecret(LOGIN_USER_PREFIX + database);
        if (user == null) {
            return null;
        }
        return new PasswordStore.LoginId(database, new String(user));
    }

    @Override
    public char[] getLoginSecret(String database) throws IOException {
        this.assertInitialized();
        return this.secretHash.getSecret(LOGIN_PW_PREFIX + database);
    }

    @Override
    public boolean deleteLogin(String db) {
        this.assertInitialized();
        if (this.secretHash.getSecret(LOGIN_USER_PREFIX + db) == null && this.secretHash.getSecret(LOGIN_PW_PREFIX + db) == null) {
            return false;
        }
        this.secretHash.deleteSecret(LOGIN_USER_PREFIX + db);
        this.secretHash.deleteSecret(LOGIN_PW_PREFIX + db);
        this.modified = true;
        return true;
    }

    @Override
    public boolean setPassphrase(char[] passphrase) throws IOException {
        this.assertInitialized();
        if (passphrase != null) {
            throw new UnsupportedOperationException("Passphrases are not supported");
        }
        return true;
    }

    @Override
    public void save() throws IOException {
        this.assertInitialized();
        if (this.secretHash == null) {
            throw new IllegalStateException("Password store has not been initialized");
        }
        if (this.modified) {
            this.secretHash.write(this.storeLocation);
            this.modified = false;
        }
    }

    @Override
    public void discard() {
        if (this.secretHash != null) {
            this.secretHash.discard();
        }
    }

    @Override
    public boolean requiresPassphrase() throws IOException {
        return false;
    }

    @Override
    public boolean isValidPassphrase(char[] passphrase) {
        return passphrase == null;
    }

    @Override
    public boolean exists() throws IOException {
        return this.storeLocation.exists();
    }

    private void assertNotInitialized() {
        if (this.secretHash != null) {
            throw new IllegalStateException("Password store already initialized");
        }
    }

    private void assertInitialized() {
        if (this.secretHash == null) {
            throw new IllegalStateException("Password store not yet initialized");
        }
    }

    private static final class SecretHash {
        private HashMap<String, char[]> secretData = new HashMap();
        private static final String PASSWORD_STORE_KEY = "Password Store:";

        SecretHash() {
        }

        private Iterator<String> aliases() {
            return this.secretData.keySet().iterator();
        }

        private void setSecret(String alias, char[] secret) {
            this.secretData.put(alias, Arrays.copyOf(secret, secret.length));
        }

        private char[] getSecret(String alias) {
            char[] secret = this.secretData.get(alias);
            if (secret == null) {
                return null;
            }
            return Arrays.copyOf(secret, secret.length);
        }

        private boolean containsAlias(String alias) {
            return this.secretData.containsKey(alias);
        }

        private boolean deleteSecret(String alias) {
            char[] secret = this.secretData.remove(alias);
            if (secret == null) {
                return false;
            }
            Arrays.fill(secret, ' ');
            return true;
        }

        private void discard() {
            this.discardSecretData();
        }

        private void discardSecretData() {
            for (char[] secret : this.secretData.values()) {
                Arrays.fill(secret, ' ');
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        synchronized void read(File f) throws IOException {
            this.discardSecretData();
            BufferedReader br = new BufferedReader(new FileReader(f));
            String keyLine = br.readLine();
            try {
                String input;
                if (keyLine == null || !keyLine.startsWith(PASSWORD_STORE_KEY)) {
                    throw new PasswordStoreException("The file does not appear to contain a password store");
                }
                while ((input = br.readLine()) != null) {
                    int firstEqual = input.indexOf("=");
                    if (firstEqual == -1 || firstEqual + 1 >= input.length()) continue;
                    String alias = input.substring(0, firstEqual).trim();
                    char[] secret = input.substring(firstEqual + 1, input.length()).trim().toCharArray();
                    this.secretData.put(alias, secret);
                }
            }
            finally {
                br.close();
            }
        }

        synchronized void write(File f) throws IOException {
            PrintWriter writer = new PrintWriter(f);
            writer.println(PASSWORD_STORE_KEY);
            Set<String> keys = this.secretData.keySet();
            for (String alias : keys) {
                char[] secret = this.secretData.get(alias);
                writer.print(alias + "=");
                writer.println(secret);
            }
            writer.close();
        }
    }
}

