/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.util;

import com.sleepycat.util.PackedInteger;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.reflect.Array;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.concurrent.TimeUnit;
import oracle.kv.impl.util.FastExternalizable;
import oracle.kv.impl.util.ObjectUtil;

public class SerializationUtil {
    public static final String MAX_SEQUENCE_LENGTH_PROPERTY = "oracle.kv.serial.max.sequence.length";
    private static final int DEFAULT_MAX_SEQUENCE_LENGTH = Integer.MAX_VALUE;
    public static volatile int maxSequenceLength = Integer.getInteger("oracle.kv.serial.max.sequence.length", Integer.MAX_VALUE);
    private static final Charset utf8 = Charset.forName("UTF-8");
    public static final String EMPTY_STRING = new String();
    public static final byte[] EMPTY_BYTES = new byte[0];
    private static final TimeUnit[] TIMEUNIT_VALUES = TimeUnit.values();
    private static final int LOCAL_BUFFER_SIZE = 16;
    private static final ThreadLocal<byte[]> localBuffer = ThreadLocal.withInitial(() -> new byte[16]);

    public static <T> T getObject(byte[] bytes, Class<T> oclass) {
        if (bytes == null) {
            return null;
        }
        ObjectInputStream ois = null;
        try {
            ois = new ObjectInputStream(new ByteArrayInputStream(bytes));
            Object object = ois.readObject();
            return (T)object;
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Exception deserializing object: " + oclass.getName(), ioe);
        }
        catch (ClassNotFoundException cnfe) {
            throw new IllegalStateException("Exception deserializing object: " + oclass.getName(), cnfe);
        }
        finally {
            if (ois != null) {
                try {
                    ois.close();
                }
                catch (IOException ioe) {
                    throw new IllegalStateException("Exception closing deserializing stream", ioe);
                }
            }
        }
    }

    public static byte[] getBytes(Object object) {
        ObjectOutputStream oos = null;
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            oos = new ObjectOutputStream(baos);
            oos.writeObject(object);
            oos.close();
            byte[] byArray = baos.toByteArray();
            return byArray;
        }
        catch (IOException ioe) {
            throw new IllegalStateException("Exception serializing object: " + object.getClass().getName(), ioe);
        }
        finally {
            if (oos != null) {
                try {
                    oos.close();
                }
                catch (IOException ioe) {
                    throw new IllegalStateException("Exception closing serializing stream", ioe);
                }
            }
        }
    }

    public static int readPackedInt(DataInput in) throws IOException {
        byte[] bytes = localBuffer.get();
        in.readFully(bytes, 0, 1);
        int len = PackedInteger.getReadIntLength(bytes, 0);
        if (len > 5) {
            throw new IOException("Invalid packed int length: " + len);
        }
        try {
            in.readFully(bytes, 1, len - 1);
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Invalid packed int", e);
        }
        return PackedInteger.readInt(bytes, 0);
    }

    public static void writePackedInt(DataOutput out, int value) throws IOException {
        byte[] buf = localBuffer.get();
        int offset = PackedInteger.writeInt(buf, 0, value);
        out.write(buf, 0, offset);
    }

    public static long readPackedLong(DataInput in) throws IOException {
        byte[] bytes = localBuffer.get();
        in.readFully(bytes, 0, 1);
        int len = PackedInteger.getReadLongLength(bytes, 0);
        if (len > 9) {
            throw new IOException("Invalid packed long length: " + len);
        }
        try {
            in.readFully(bytes, 1, len - 1);
        }
        catch (IndexOutOfBoundsException e) {
            throw new IOException("Invalid packed long", e);
        }
        return PackedInteger.readLong(bytes, 0);
    }

    public static void writePackedLong(DataOutput out, long value) throws IOException {
        byte[] buf = localBuffer.get();
        int offset = PackedInteger.writeLong(buf, 0, value);
        out.write(buf, 0, offset);
    }

    public static String readString(DataInput in, short serialVersion) throws IOException {
        return serialVersion >= 14 ? SerializationUtil.readStdUTF8String(in) : SerializationUtil.readJavaString(in);
    }

    public static String readNonNullString(DataInput in, short serialVersion) throws IOException {
        String result = SerializationUtil.readString(in, serialVersion);
        if (result == null) {
            throw new IOException("Found null value for non-null string");
        }
        return result;
    }

    public static void writeString(DataOutput out, short serialVersion, String value) throws IOException {
        if (serialVersion >= 14) {
            SerializationUtil.writeStdUTF8String(out, value);
        } else {
            SerializationUtil.writeJavaString(out, value);
        }
    }

    public static void writeNonNullString(DataOutput out, short serialVersion, String value) throws IOException {
        ObjectUtil.checkNull("value", value);
        SerializationUtil.writeString(out, serialVersion, value);
    }

    private static String readJavaString(DataInput in) throws IOException {
        int len = SerializationUtil.readPackedInt(in);
        if (len < 0) {
            return null;
        }
        if (len == 0) {
            return EMPTY_STRING;
        }
        SerializationUtil.checkMaxSequenceLength(len);
        byte[] bytes = SerializationUtil.getByteBuffer(len);
        in.readFully(bytes, 0, len);
        try {
            return SerializationUtil.utfToString(bytes, len);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new IOException("Invalid UTF-8 encoding", e);
        }
    }

    private static void checkMaxSequenceLength(int length) throws IOException {
        if (length > maxSequenceLength) {
            throw new IOException("Sequence length " + length + " is greater than maximum sequence length " + maxSequenceLength);
        }
    }

    private static byte[] getByteBuffer(int len) {
        if (len > 16) {
            return new byte[len];
        }
        return localBuffer.get();
    }

    private static void writeJavaString(DataOutput out, String value) throws IOException {
        if (value == null) {
            SerializationUtil.writePackedInt(out, -1);
            return;
        }
        int utfLength = SerializationUtil.getUTFLength(value);
        SerializationUtil.writePackedInt(out, utfLength);
        if (utfLength > 0) {
            byte[] bytes = SerializationUtil.stringToUTF(value, utfLength);
            out.write(bytes, 0, utfLength);
        }
    }

    private static int getUTFLength(String s) {
        int len = 0;
        int length = s.length();
        for (int i = 0; i < length; ++i) {
            char c = s.charAt(i);
            if (c >= '\u0001' && c <= '\u007f') {
                ++len;
                continue;
            }
            if (c > '\u07ff') {
                len += 3;
                continue;
            }
            len += 2;
        }
        return len;
    }

    private static byte[] stringToUTF(String s, int utfLength) {
        char c;
        int i;
        int length = s.length();
        assert (length > 0);
        byte[] bytes = SerializationUtil.getByteBuffer(utfLength);
        int byteOffset = 0;
        for (i = 0; i < length && (c = s.charAt(i)) >= '\u0001' && c <= '\u007f'; ++i) {
            bytes[byteOffset++] = (byte)c;
        }
        while (i < length) {
            if ((c = s.charAt(i++)) >= '\u0001' && c <= '\u007f') {
                bytes[byteOffset++] = (byte)c;
                continue;
            }
            if (c > '\u07ff') {
                bytes[byteOffset++] = (byte)(0xE0 | c >> 12 & 0xF);
                bytes[byteOffset++] = (byte)(0x80 | c >> 6 & 0x3F);
                bytes[byteOffset++] = (byte)(0x80 | c >> 0 & 0x3F);
                continue;
            }
            bytes[byteOffset++] = (byte)(0xC0 | c >> 6 & 0x1F);
            bytes[byteOffset++] = (byte)(0x80 | c >> 0 & 0x3F);
        }
        return bytes;
    }

    private static String utfToString(byte[] bytes, int len) throws IOException {
        int char1;
        int count;
        if (len == 0) {
            return EMPTY_STRING;
        }
        char[] chars = new char[len];
        int charArrayCount = 0;
        for (count = 0; count < len && (char1 = bytes[count] & 0xFF) <= 127; ++count) {
            chars[charArrayCount++] = (char)char1;
        }
        block6: while (count < len) {
            char1 = bytes[count++] & 0xFF;
            switch (char1 >> 4) {
                case 0: 
                case 1: 
                case 2: 
                case 3: 
                case 4: 
                case 5: 
                case 6: 
                case 7: {
                    chars[charArrayCount++] = (char)char1;
                    continue block6;
                }
                case 12: 
                case 13: {
                    byte char2 = bytes[count++];
                    if ((char2 & 0xC0) != 128) {
                        throw new IOException("Invalid UTF-8 encoding");
                    }
                    chars[charArrayCount++] = (char)((char1 & 0x1F) << 6 | char2 & 0x3F);
                    continue block6;
                }
                case 14: {
                    byte char2 = bytes[count++];
                    byte char3 = bytes[count++];
                    if ((char2 & 0xC0) != 128 || (char3 & 0xC0) != 128) {
                        throw new IOException("Invalid UTF-8 encoding");
                    }
                    chars[charArrayCount++] = (char)((char1 & 0xF) << 12 | (char2 & 0x3F) << 6 | (char3 & 0x3F) << 0);
                    continue block6;
                }
            }
            throw new IOException("Invalid UTF-8 encoding");
        }
        return new String(chars, 0, charArrayCount);
    }

    private static void writeStdUTF8String(DataOutput out, String value) throws IOException {
        if (value == null) {
            SerializationUtil.writePackedInt(out, -1);
            return;
        }
        ByteBuffer buffer = utf8.encode(value);
        int length = buffer.limit();
        SerializationUtil.writePackedInt(out, length);
        if (length > 0) {
            out.write(buffer.array(), 0, length);
        }
    }

    private static String readStdUTF8String(DataInput in) throws IOException {
        int length = SerializationUtil.readPackedInt(in);
        if (length < 0) {
            return null;
        }
        if (length == 0) {
            return EMPTY_STRING;
        }
        SerializationUtil.checkMaxSequenceLength(length);
        byte[] bytes = SerializationUtil.getByteBuffer(length);
        in.readFully(bytes, 0, length);
        return utf8.decode(ByteBuffer.wrap(bytes, 0, length)).toString();
    }

    public static int readSequenceLength(DataInput in) throws IOException {
        int result = SerializationUtil.readPackedInt(in);
        if (result < -1) {
            throw new IOException("Invalid sequence length: " + result);
        }
        SerializationUtil.checkMaxSequenceLength(result);
        return result;
    }

    public static int readNonNullSequenceLength(DataInput in) throws IOException {
        int length = SerializationUtil.readSequenceLength(in);
        if (length == -1) {
            throw new IOException("Read null length for non-null sequence");
        }
        return length;
    }

    public static void writeCollectionLength(DataOutput out, Collection<?> collection) throws IOException {
        SerializationUtil.writeSequenceLength(out, collection != null ? collection.size() : -1);
    }

    public static void writeArrayLength(DataOutput out, Object array) throws IOException {
        SerializationUtil.writeSequenceLength(out, array != null ? Array.getLength(array) : -1);
    }

    public static void writeSequenceLength(DataOutput out, int length) throws IOException {
        if (length < -1) {
            throw new IllegalArgumentException("Invalid sequence length: " + length);
        }
        SerializationUtil.writePackedInt(out, length);
    }

    public static void writeNonNullSequenceLength(DataOutput out, int length) throws IOException {
        if (length < 0) {
            throw new IllegalArgumentException("Invalid non-null sequence length: " + length);
        }
        SerializationUtil.writePackedInt(out, length);
    }

    public static byte[] readByteArray(DataInput in) throws IOException {
        int len = SerializationUtil.readSequenceLength(in);
        if (len == -1) {
            return null;
        }
        if (len == 0) {
            return EMPTY_BYTES;
        }
        byte[] array = new byte[len];
        in.readFully(array);
        return array;
    }

    public static void writeByteArray(DataOutput out, byte[] array) throws IOException {
        int length = array == null ? -1 : Array.getLength(array);
        SerializationUtil.writeArrayLength(out, array);
        if (length > 0) {
            out.write(array);
        }
    }

    public static byte[] readNonNullByteArray(DataInput in) throws IOException {
        byte[] array = SerializationUtil.readByteArray(in);
        if (array == null) {
            throw new IOException("Read unexpected null array");
        }
        return array;
    }

    public static void writeNonNullByteArray(DataOutput out, byte[] array) throws IOException {
        ObjectUtil.checkNull("array", array);
        SerializationUtil.writeByteArray(out, array);
    }

    public static void writeNonNullCollection(DataOutput out, short serialVersion, Collection<? extends FastExternalizable> collection) throws IOException {
        ObjectUtil.checkNull("collection", collection);
        SerializationUtil.writeNonNullSequenceLength(out, collection.size());
        for (FastExternalizable fastExternalizable : collection) {
            fastExternalizable.writeFastExternal(out, serialVersion);
        }
    }

    public static void writeNonNullArray(DataOutput out, short serialVersion, FastExternalizable[] array) throws IOException {
        ObjectUtil.checkNull("array", array);
        SerializationUtil.writeNonNullSequenceLength(out, array.length);
        for (FastExternalizable e : array) {
            e.writeFastExternal(out, serialVersion);
        }
    }

    public static byte[] readByteArrayOldShortLength(DataInput in, short serialVersion) throws IOException {
        if (serialVersion >= 14) {
            return SerializationUtil.readByteArray(in);
        }
        short len = in.readShort();
        if (len == -1) {
            return null;
        }
        if (len == 0) {
            return EMPTY_BYTES;
        }
        SerializationUtil.checkMaxSequenceLength(len);
        byte[] array = new byte[len];
        in.readFully(array);
        return array;
    }

    public static void writeByteArrayOldShortLength(DataOutput out, short serialVersion, byte[] array) throws IOException {
        if (serialVersion >= 14) {
            SerializationUtil.writeByteArray(out, array);
        } else if (array == null) {
            out.writeShort(-1);
        } else {
            out.writeShort(array.length);
            out.write(array);
        }
    }

    public static byte[] readNonNullByteArrayOldShortLength(DataInput in, short serialVersion) throws IOException {
        byte[] array = SerializationUtil.readByteArrayOldShortLength(in, serialVersion);
        if (array == null) {
            throw new IOException("Expected non-null array");
        }
        return array;
    }

    public static void writeNonNullByteArrayOldShortLength(DataOutput out, short serialVersion, byte[] array) throws IOException {
        ObjectUtil.checkNull("array", array);
        SerializationUtil.writeByteArrayOldShortLength(out, serialVersion, array);
    }

    public static <T extends FastExternalizable> void writeFastExternalOrNull(DataOutput out, short serialVersion, T object) throws IOException {
        if (object != null) {
            out.writeBoolean(true);
            object.writeFastExternal(out, serialVersion);
        } else {
            out.writeBoolean(false);
        }
    }

    public static void writeTimeUnit(DataOutput out, TimeUnit timeUnit) throws IOException {
        ObjectUtil.checkNull("timeUnit", timeUnit);
        SerializationUtil.writePackedInt(out, timeUnit.ordinal());
    }

    public static TimeUnit readTimeUnit(DataInput in) throws IOException {
        int ordinal = SerializationUtil.readPackedInt(in);
        try {
            return TIMEUNIT_VALUES[ordinal];
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new IOException("Illegal TimeUnit ordinal: " + ordinal);
        }
    }
}

