#!/bin/bash
#
# orakafka_distro_install.sh
#
# Copyright (c) 2020, Oracle and/or its affiliates. All rights reserved.
#
#    NAME
#      orakafka_distro_install.sh 
#
#    DESCRIPTION
#    Install orakafka kit in ora_kafka_home directory 
#
#    Usage:
#    orakafka_distro_install.sh -p <ora-kafka-home> [-a <app-data-home>] [-h]
#    where options are:
#     -p <ora-kafka-home> - absolute path of ora_kafka_home/ directory 
#                           where the product is installed. 
#                           The ora_kafka_home/ directory is created if it
#                           does not exist
#
#                         By default 
#                          <ora_kafka_home>/orakafka contains kit binaries
#                           and
#                          <ora_kafka_home>/app_data contains instance data
#                          such as cluster conf,scripts and logs
#
#     -a <app-data-home>  - absolute path of app_data/ directory 
#                           default: <ora-kafka-home>/app_data
#
#                           Specify this option for RAC databases and/or to 
#                           override the default location of app_data/directory
#                           For RAC databases this must be a cluster filesystem
#                           path.
#
#                           The app_data/ directories are created if they
#                           do not exist but are never overwritten.
#                           To reinstall the app_data/ directory in the same 
#                           location, you must first manually move the
#                           app_data/ directory to a different location and
#                           then run this script again.
#
#     -h                  - Display usage
# 
#    NOTES
#      For debugging, set environment variable ORAKAFKA_CONFIG_DEBUG to true.
#      export ORAKAFKA_CONFIG_DEBUG=true
#
#



function debug {
  if [[ "${ORAKAFKA_CONFIG_DEBUG}" == true ]]; then
    # if a debug flag is set, echo the message
    if [[ ! -z "$1" ]]; then
      echo "DEBUG: $1"
    fi
  fi
}

function printstepmsg {
  MSG="$1"  
  printf "%s\n" "$MSG"
}

function printstatus {
  ret_code=$1
  step=$2
  if [[ "$ret_code" -eq 0 ]]; then
    echo "$step completed."
  fi
}


function failuremsg {
  printf "FAILURE: %s\n" "$1" >&2
}

function successmsg {
  printf "SUCCESS: %s\n" "$1"
}

function printheader {
  printf "\n%s\n" "$1"
  printf "%s\n" "--------------------------------------------------------------"
}


function usage {
  echo "Usage: ${SCRIPT_NAME} -p <ora-kafka-home> [-a <app-data-home>] [-h]" 
  echo "where options are:"
  echo "-p <ora-kafka-home> - absolute path of ora_kafka_home/ directory" 
  echo "                      where the product is installed" 
  echo "                      The ora_kafka_home/ directory is created if it does not exist"
  echo "                           "
  echo "                       By default" 
  echo "                        <ora_kafka_home>/orakafka contains kit binaries"
  echo "                        and"
  echo "                        <ora_kafka_home>/app_data contains instance data"
  echo "                        such as cluster conf,scripts and logs"
  echo "                           "
  echo "-a <app-data-home>  - absolute path of app_data/ directory" 
  echo "                      default: <ora-kafka-home>/app_data"
  echo "                           "
  echo "                      Specify this option for RAC databases or to "
  echo "                      override the default location of app_data/ directory"
  echo "                      For RAC databases this must be a cluster filesystem path"
  echo "                           "
  echo "                      The app_data/ directory is created if it does not exist"
  echo "                      but is never overwritten."
  echo "                      To reinstall the app_data/ directory in the same location,"
  echo "                      you must first manually move the app_data/ directory" 
  echo "                      to a different location and then run this script again."
  echo "                           "
  echo "-h                  - Display usage"
  exit 1
}

function cleanup {
  printstepmsg "$STEP failed."
  PRODUCT_HOME="$1"
  SUFFIX="$2"
  KIT_HOME="${PRODUCT_HOME}"/orakafka
  KIT_HOME_BAK="${PRODUCT_HOME}"/orakafka_bak

  echo "removing ${KIT_HOME}.."
  if [[ -d "${KIT_HOME}" ]]; then
    rm -rf "${KIT_HOME}"
  fi  

  if [[ -d  "${KIT_HOME_BAK}/orakafka-${SUFFIX}" ]]; then
      echo "restoring from ${KIT_HOME_BAK} to $KIT_HOME"
      mv "${KIT_HOME_BAK}/orakafka-${SUFFIX}" "${PRODUCT_HOME}/orakafka"
  fi  
  
  exit 1
}

function failstep {
  printstepmsg "$STEP failed."
  exit 1
}


#
# create the ora_kafka_home and set  PRODUCT_HOME global var
#
function create_product_home { 
  printheader " ${STEP}:"

  local PRODUCT_ROOT="$1"
  debug "Input arg: PRODUCT_ROOT=[${PRODUCT_ROOT}]"

  local length=${#PRODUCT_ROOT}
  local last_char=${PRODUCT_ROOT:length-1:1}
  #remove trailing slash '/'
  [[ $last_char == "/" ]] && PRODUCT_ROOT=${PRODUCT_ROOT:0:length-1}; :
  debug "after trim last_char PRODUCT_ROOT=[${PRODUCT_ROOT}]"


  #
  # check if base directory is ora_kafka_home
  #
  local PRODUCT_ROOT_BASENAME=$(basename "${PRODUCT_ROOT}")
  debug "PRODUCT_ROOT_BASENAME=[${PRODUCT_ROOT_BASENAME}]"
  local PRODUCT_ROOT_DIRNAME=$(dirname "${PRODUCT_ROOT}")
  debug "PRODUCT_ROOT_DIRNAME=[${PRODUCT_ROOT_DIRNAME}]"

  if [ "$PRODUCT_ROOT_BASENAME" != "ora_kafka_home" ] &&  [ -d "${PRODUCT_ROOT}" ];
  then
      #remove any symbolic links or relative paths
      PRODUCT_ROOT=$(cd -P -- "${PRODUCT_ROOT}" && pwd -P)
  elif [ "$PRODUCT_ROOT_BASENAME" == "ora_kafka_home" ] &&  [ -d "${PRODUCT_ROOT_DIRNAME}" ];
  then    
      #remove any symbolic links or relative paths
      PRODUCT_ROOT=$(cd -P -- "${PRODUCT_ROOT_DIRNAME}" && pwd -P)
  else  
      echo "ora-kafka-home directory: [$PRODUCT_ROOT] does not exist."
      echo "create directory and run ${SCRIPT_NAME} again"
      exit 1
  fi 
  debug "Physical path of PRODUCT_ROOT=[${PRODUCT_ROOT}]"
  PRODUCT_HOME=${PRODUCT_ROOT}/ora_kafka_home
  debug "PRODUCT_HOME=[${PRODUCT_HOME}]"


  if  [[ ! -d "${PRODUCT_ROOT}" ]];  then
  
     echo "ora-kafka-home-root directory: [$PRODUCT_ROOT] does not exist."
     echo "create directory and run ${SCRIPT_NAME} again"
     exit 1
  fi

  #
  # create ora_kafka_home/ if it does not exist
  #
  if [[ ! -d "$PRODUCT_HOME" ]]; then
    mkdir "${PRODUCT_HOME}"
    chmod 755 "${PRODUCT_HOME}"
  else 
    echo ".....${PRODUCT_HOME} already exists.."
    local KIT_HOME="$PRODUCT_HOME"/orakafka
    if [[ -d "$KIT_HOME" ]]; then
      printheader " ${KIT_HOME} directory already exists:"
      printstepmsg ".....checking kit version in ${KIT_HOME}"

      #this is used for info/error messages

      local DISTRO_KIT_VER_STR=$(cat "$DISTRO_HOME"/kit_version.txt)
      local PRODUCT_KIT_VER_STR=$(cat "$KIT_HOME"/kit_version.txt)

      #the numeric version# is used for comparison
      #kit_version.txt contains version number - format: n.n.n 
      #example: 1.2.0
      local DISTRO_KIT_VER_NUM=$(sed 's/\.//g' "$DISTRO_HOME"/kit_version.txt)
      local PRODUCT_KIT_VER_NUM=$(sed 's/\.//g' "$KIT_HOME"/kit_version.txt)


      if [[ "$DISTRO_KIT_VER_NUM" -eq "$PRODUCT_KIT_VER_NUM" ]]; then
        # case 1: product version is already installed
        printstepmsg ".....DISTRO_KIT_VERSION=$DISTRO_KIT_VER_STR same as PRODUCT_KIT_VERSION=$PRODUCT_KIT_VER_STR" 
        echo "Product already exists in ${KIT_HOME}"
        echo "To reinstall, remove ${KIT_HOME} directory and run ${SCRIPT_NAME} again"
        exit 0
      elif [[ "$DISTRO_KIT_VER_NUM" -lt "$PRODUCT_KIT_VER_NUM" ]]; then
        # case 2: downgrade not supported
        printstepmsg ".....DISTRO_KIT_VERSION=$DISTRO_KIT_VER_STR less than PRODUCT_KIT_VERSION=$PRODUCT_KIT_VER_STR" 
        echo "...Install error:Downgrade is not supported"
        exit 0
      else
        # case 3: upgrade supported
        printstepmsg ".....DISTRO_KIT_VERSION=$DISTRO_KIT_VER_STR greater than PRODUCT_KIT_VERSION=$PRODUCT_KIT_VER_STR" 
        echo ".....proceeding with install"
      fi 
    fi  
  fi 
}

#
# create the app_data/ and set APP_DATA_HOME global var
#
function create_app_data_home {
  printheader " ${STEP}:"

  local PRODUCT_HOME="$1"
  local APP_DATA_ROOT="$2"
  debug "PRODUCT_HOME=[${PRODUCT_HOME}]"
  debug "APP_DATA_ROOT=[${APP_DATA_ROOT}]"


  local length=${#APP_DATA_ROOT}
  local last_char=${APP_DATA_ROOT:length-1:1}
  #remove trailing slash '/'
  [[ $last_char == "/" ]] && APP_DATA_ROOT=${APP_DATA_ROOT:0:length-1}; :
  debug "after trim last_char APP_DATA_ROOT=[${APP_DATA_ROOT}]"


  #
  # check if base directory is app_data/
  #
  local APP_DATA_ROOT_BASENAME=$(basename "${APP_DATA_ROOT}")
  debug "APP_DATA_ROOT_BASENAME=[${APP_DATA_ROOT_BASENAME}]"
  local APP_DATA_ROOT_DIRNAME=$(dirname "${APP_DATA_ROOT}")
  debug "APP_DATA_ROOT_DIRNAME=[${APP_DATA_ROOT_DIRNAME}]"

  if [ "$APP_DATA_ROOT_BASENAME" != "app_data" ] &&  [ -d "${APP_DATA_ROOT}" ];
  then
      #remove any symbolic links or relative paths
      APP_DATA_ROOT=$(cd -P -- "${APP_DATA_ROOT}" && pwd -P)
  elif [ "$APP_DATA_ROOT_BASENAME" == "app_data" ] &&  [ -d "${APP_DATA_ROOT_DIRNAME}" ];
  then    
      #remove any symbolic links or relative paths
      APP_DATA_ROOT=$(cd -P -- "${APP_DATA_ROOT_DIRNAME}" && pwd -P)
  else  
      echo "root directory: [$APP_DATA_ROOT] does not exist."
      echo "create directory and run ${SCRIPT_NAME} again"
      exit 1
  fi 
  debug "Physical path of APP_DATA_ROOT=[${APP_DATA_ROOT}]"
  APP_DATA_HOME=${APP_DATA_ROOT}/app_data
  debug "APP_DATA_HOME=[${APP_DATA_HOME}]"


  # check if root_dir of app_data exists
  if  [[ ! -d "$APP_DATA_ROOT" ]]; then
     echo "app-data-root directory: $APP_DATA_ROOT does not exist."
     echo "create directory and run ${SCRIPT_NAME} again"
     exit 0
  fi

  if [ -d "${PRODUCT_HOME}"/app_data ] && 
     [ "${APP_DATA_HOME}" != "${PRODUCT_HOME}"/app_data ]; 
  then
    printstepmsg "......WARNING: PRODUCT_HOME/app_data=[${PRODUCT_HOME}/app_data] already exists"
    printstepmsg "......WARNING: Different APP_DATA_HOME=${APP_DATA_HOME} specified"
    printstepmsg "......Starting setup of APP_DATA_HOME=${APP_DATA_HOME} "
  fi

  # create APP_DATA_HOME 
  if [[ ! -d "${APP_DATA_HOME}" ]]; then
    printstepmsg "..... creating ${APP_DATA_HOME} and subdirectories"
    mkdir "${APP_DATA_HOME}"
    mkdir "${APP_DATA_HOME}/clusters"
    mkdir "${APP_DATA_HOME}/logs"
    mkdir "${APP_DATA_HOME}/scratch"
    mkdir "${APP_DATA_HOME}/scripts"
    chmod -R 755 "${APP_DATA_HOME}"
  else
    printstepmsg ".....${APP_DATA_HOME} already exists"
    if [[ ! -d "${APP_DATA_HOME}/clusters" ]]; then
      printstepmsg "........creating ${APP_DATA_HOME}/clusters"
      mkdir "${APP_DATA_HOME}/clusters"
      chmod -R 755 "${APP_DATA_HOME}/clusters"
    else  
      printstepmsg "........${APP_DATA_HOME}/clusters already exists"
    fi  

    if [[ ! -d "${APP_DATA_HOME}/logs" ]]; then
      printstepmsg "..... creating ${APP_DATA_HOME}/logs"
      mkdir "${APP_DATA_HOME}/logs"
      chmod -R 755 "${APP_DATA_HOME}/logs"
    else  
      printstepmsg "........${APP_DATA_HOME}/logs already exists"
    fi  

    if [[ ! -d "${APP_DATA_HOME}/scripts" ]]; then
      printstepmsg "........creating ${APP_DATA_HOME}/scripts"
      mkdir "${APP_DATA_HOME}/scripts"
      chmod -R 755 "${APP_DATA_HOME}/scripts"
    else  
      printstepmsg "........${APP_DATA_HOME}/scripts already exists"
    fi  

    if [[ ! -d "${APP_DATA_HOME}/scratch" ]]; then
      printstepmsg "........creating ${APP_DATA_HOME}/scratch"
      mkdir "${APP_DATA_HOME}/scratch"
      chmod -R 755 "${APP_DATA_HOME}/scratch"
    else  
      printstepmsg "........${APP_DATA_HOME}/scratch already exists"
    fi  
  fi  
  
  #
  # write the PRODUCT_KIT_HOME into APP_DATA_HOME/scripts/configure_kit_home.sh
  #
  # The preprocessor script must be able to find jlib/ libs in PRODUCT_HOME 
  # and APP_DATA_HOME can be outside the PRODUCT_HOME
  # this also gets us out of using relative paths
  local CONF_KIT_HOME_SCRIPT="${APP_DATA_HOME}/scripts/configure_kit_home.sh"
  echo "# generated by orakafka_distro_install.sh" > "${CONF_KIT_HOME_SCRIPT}"
  echo "export PRODUCT_KIT_HOME=${PRODUCT_HOME}/orakafka" >> ${CONF_KIT_HOME_SCRIPT}
  chmod 755 ${CONF_KIT_HOME_SCRIPT}

  debug " contents of ${CONF_KIT_HOME_SCRIPT}"
  debug "$(cat ${CONF_KIT_HOME_SCRIPT})"
  printstepmsg "......Generated CONF_KIT_HOME_SCRIPT=${CONF_KIT_HOME_SCRIPT}"

  #
  # write the value of APP_DATA_HOME into PRODUCT_HOME 
  # 
  # APP_DATA_HOME can be outside the PRODUCT_HOME 
  # orakafka.sh and other scripts need to know where APP_DATA_HOME is 
  # to write logs and scratch/ directories. 
  # 
  local CONF_APP_DATA_HOME_SCRIPT="${PRODUCT_HOME}/configure_app_data_home.sh"
  echo "# generated by orakafka_distro_install.sh" > "${CONF_APP_DATA_HOME_SCRIPT}"
  echo "export APP_DATA_HOME=${APP_DATA_HOME}" >> ${CONF_APP_DATA_HOME_SCRIPT}
  chmod 755 ${CONF_APP_DATA_HOME_SCRIPT}

  debug " contents of ${CONF_APP_DATA_HOME_SCRIPT}"
  debug "$(cat ${CONF_APP_DATA_HOME_SCRIPT})"
  printstepmsg "......Generated CONF_APP_DATA_HOME_SCRIPT=${CONF_APP_DATA_HOME_SCRIPT}"

}

#
# Backup ora_kafka_home/orakafka -> ora_kafka_home/orakafka_bak/orakafka-<DATETIME>
# unzip DISTRO_HOME/orakafka.zip -> ora_kafka_home/orakafka
#
function unzip_kit {
  local CURRENT_TIME=$(date "+%Y.%m.%d-%H.%M.%S")
  local PRODUCT_HOME="$1"
  local DISTRO_HOME="$2"
  local KIT_HOME="$PRODUCT_HOME"/orakafka
  local KIT_HOME_BAK="$PRODUCT_HOME"/orakafka_bak


  printheader " ${STEP}:"
  printstepmsg ".....checking for existing binaries in $KIT_HOME"

  if [[ -d "$KIT_HOME" ]]; then
   local DISTRO_KIT_VERSION=$(cat "$DISTRO_HOME"/kit_version.txt)
   local PRODUCT_KIT_VERSION=$(cat "$PRODUCT_HOME"/orakafka/kit_version.txt)
   if [[ "$DISTRO_KIT_VERSION" == "$PRODUCT_KIT_VERSION" ]]; then
     printstepmsg ".....DISTRO_KIT_VERSION=$DISTRO_KIT_VERSION same as PRODUCT_KIT_VERSION=$PRODUCT_KIT_VERSION" 
     printstepmsg ".....skipping backup; removing ${KIT_HOME}"
     rm -rf "${KIT_HOME}"
   else
     printstepmsg ".....moving existing binaries ${KIT_HOME_BAK}/orakafka-$CURRENT_TIME/"
     if [[ ! -d "$KIT_HOME_BAK" ]]; then
       printstepmsg ".....creating $KIT_HOME_BAK"
       mkdir ${KIT_HOME_BAK}
       chmod 755 "${KIT_HOME_BAK}"
     fi 
     mv  "${KIT_HOME}" "${KIT_HOME_BAK}"/orakafka-$CURRENT_TIME
   fi  
  fi

  trap "cleanup ${PRODUCT_HOME} $CURRENT_TIME"  INT ERR TERM
  printstepmsg ".....unzip kit into ${KIT_HOME}"
  unzip -o "$DISTRO_HOME"/orakafka.zip -d "${PRODUCT_HOME}"
}


# get the name of this script from BASH_SOURCE 
# or from $0 (if BASH_SOURCE is null or empty)
SCRIPT_NAME="${BASH_SOURCE:-$0}"
#
# get the current directory without symbolic links
#
# cd -P: The -P option means to use the physical directory structure 
#        instead of following symbolic links
#    '--' indicates end of options to the builtin command
#         See Bash Ref Manual: Shell Builtin commands for more info on '--'
# pwd -P: The -P ensures that the printed pathname will not contain any 
#         symbolic links
#
DISTRO_HOME=$(cd -P -- "$(dirname -- "$SCRIPT_NAME")" && pwd -P)

debug "DISTRO_HOME=${DISTRO_HOME}"
debug "SCRIPT_NAME=${SCRIPT_NAME}"


if [[ $# -eq 0 ]]; then
  usage
fi

debug "Arguments passed to ${SCRIPT_NAME}: $*"

# set any defaults

while getopts ":p:a:h" opt;
do
  case "$opt" in
   p) PRODUCT_ROOT=$OPTARG;;
   a) APP_DATA_ROOT=$OPTARG;;
   h) usage ;;
  \?) echo "Invalid option -$OPTARG"
      usage ;;
   :) echo "Option -$OPTARG requires an argument."
      usage ;;
  esac
done
# after getopts is done, shifts away all the processed options.
shift "$(($OPTIND -1))"

if ((OPTIND == 1))
then
    echo "No options specified"
    usage
fi




#----------------------------------------------------
# create orakafka/ and set PRODUCT_HOME
# For RAC Databases: PRODUCT_HOME must be replicated on all RAC nodes
#----------------------------------------------------
trap "failstep" INT ERR TERM
STEP="Step Create Product Home:"

# 
#----------PRODUCT_HOME global var set in this function
create_product_home "$PRODUCT_ROOT" PRODUCT_HOME 
printstatus "$?" "$STEP"
echo "PRODUCT_HOME=${PRODUCT_HOME}"

#---------------------------------------------------


#----------------------------------------------------
# create app_data/ and set APP_DATA_HOME
#
# For RAC databases APP_DATA_ROOT must be a cluster
# file system directory
# By default it is created under PRODUCT_HOME/app_data
#----------------------------------------------------
trap "failstep" INT ERR TERM
STEP="Step Create app_data home:"

# -------set default value for APP_DATA_ROOT if it was not specified 
# -------Note: the default setting is not recommended for RAC databases
if [[ -z "${APP_DATA_ROOT}" ]]; then
  APP_DATA_ROOT="${PRODUCT_HOME}"
fi

# -------APP_DATA_HOME global var is set in this function
create_app_data_home "$PRODUCT_HOME" "$APP_DATA_ROOT" APP_DATA_HOME 
printstatus "$?" "$STEP"

echo "APP_DATA_HOME=${APP_DATA_HOME}"

#----------------------------------------------------


#----------------------------------------------------
# unzip orakafka.zip from DISTRO_HOME to PRODUCT_HOME 
#-----------------------------------------------------

STEP="Step unzip_kit:"
unzip_kit "$PRODUCT_HOME" "$DISTRO_HOME" 
printstatus "$?" "$STEP"


echo -e "\nSuccessfully installed orakafka kit in ${PRODUCT_HOME}"
