/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.debug;

import javax.ide.Service;
import javax.ide.spi.ProviderNotFoundException;
import javax.ide.command.Context;

/**
 * The <code>Debugger</code> can be used to start the debuggee or to 
 * retrieve command line options so the extension can start the debuggee 
 * itself.<p>
 *
 * IDE providers must provide an implementation of {@link #getClientConnector( Context )}
 * and {@link #getServerConnector(Context)}.
 */
public abstract class Debugger extends Service
{
  /**
   * Creates a server connector for a debugging session. The 
   * {@link ServerConnector} object returned can be used to start the 
   * debuggee or to retrieve command line options so the extension can start 
   * the debuggee itself.
   *
   * If the extension wants to start the the debuggee itself, it should
   * call {@link Connector#getOptions()} to retrieve the command line
   * options that it should use to start the debuggee.
   * If the extension wants to start a third party container (such as
   * Tomcat, WebLogic, Avalon based systems, properietary based
   * systems, etc.), it can ignore options that are specific to the
   * IDE's preferred container (for example, the extension may ignore the
   * values corresponding to the {@link Connector#OPTION_MAIN_CLASS}
   * key). The extension should not ignore the values corresponding to
   * the {@link Connector#OPTION_DEBUG_FIRST} and
   * {@link Connector#OPTION_DEBUG_LAST} keys, as these are dictated
   * by the IDE so that the debugger will be able connect with the
   * debuggee.
   * 
   * If the extension wants the IDE to start the debuggee, it should call
   * {@link Connector#startDebuggee(Map)}.
   * 
   * The returned Connector will implement {@link ServerConnector}. The 
   * extension writer should call {@link ServerConnector#startListening()} to 
   * start the debugger listening, and then the extension should either start 
   * the debuggee process or tell the IDE to do so by calling {@link
   * Connector#startDebuggee(Map)}. When the debuggee process is launched,
   * the listening debugger will automatically accept the connection.
   *
   * @param context the current {@link Context}.
   * @return the debugger connector
   * @exception UnsupportedOperationException if the IDE does not
   * support the debugger as server.
   */
  public abstract ServerConnector getServerConnector( Context context ) 
                                        throws UnsupportedOperationException;
  
  /**
   * Creates a client connector for a debugging session. The 
   * {@link ClientConnector} object returned can be used to start the 
   * debuggee or to retrieve command line options so the extension can start 
   * the debuggee itself.
   *
   * If the extension wants to start the the debuggee itself, it should
   * call {@link Connector#getOptions()} to retrieve the command line
   * options that it should use to start the debuggee.
   * If the extension wants to start a third party container (such as
   * Tomcat, WebLogic, Avalon based systems, properietary based
   * systems, etc.), it can ignore options that are specific to the
   * IDE's preferred container (for example, the extension may ignore the
   * values corresponding to the {@link Connector#OPTION_MAIN_CLASS}
   * key). The extension should not ignore the values corresponding to
   * the {@link Connector#OPTION_DEBUG_FIRST} and
   * {@link Connector#OPTION_DEBUG_LAST} keys, as these are dictated
   * by the IDE so that the debugger will be able connect with the
   * debuggee.
   * 
   * If the extension wants the IDE to start the debuggee, it should call
   * {@link Connector#startDebuggee(Map)}.
   * 
   * The returned Connector will implement {@link ClientConnector}. The 
   * extension writer should either start the debuggee process or tell the 
   * IDE to do so by calling {@link Connector#startDebuggee(Map)}, and then the
   * extension writer should call {@link ClientConnector#attachDebugger()} to
   * tell the debugger to attach to the debuggee.
   *
   * @param context the current {@link Context}.
  
   * @return the debugger connector
   * @exception UnsupportedOperationException if the IDE does not
   * support debugger as client.
   */
  public abstract ClientConnector getClientConnector( Context context ) 
                                        throws UnsupportedOperationException;
  

  /**
   * Get the debugger implementation for this IDE.
   * 
   * @return the debugger implementation.
   */
  public static Debugger getDebugger()
  {
    try
    {
      return (Debugger)getService( Debugger.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No debugger." );
    }
  }
}

