/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;


import javax.ide.util.Version;

/**
 * Represents an extension's dependency on some other extension.
 */
public final class ExtensionDependency 
{
  private final String _id;
  private final Version _minVersion;
  private final Version _maxVersion;

  /**
   * Constructs a dependency for any version of the specified extension id.
   * 
   * @param id an extension id. Must not be null.
   */
  public ExtensionDependency( String id )
  {
    this( id, null, null );
  }
  
  /**
   * Constructs a dependency between two versions of the specified extension id.
   * 
   * @param id an extension id. Must not be null.
   * @param min the minimum required version of the specified extension. May
   *    be null, which indicates that there is no minimum version.
   * @param max the maximum required version of the specified extension. May
   *    be null, which indicates that there is no maximum version.
   */
  public ExtensionDependency( String id, Version min, Version max )
  {
    if ( id == null )
    {
      throw new NullPointerException( "id is null" );
    }
    _id = id;
    _minVersion = min;
    _maxVersion = max;
  }
  
  /**
   * Get the id of the extension this dependency is for.
   * 
   * @return the id of an extension. Will not return null.
   */
  public String getID()
  {
    return _id;
  }
  
  /**
   * Get the minimum version of the extension this dependency is for.
   * 
   * @return the minimum version. May be null if there is no minimum required
   *    version.
   */
  public Version getMinimumVersion()
  {
    return _minVersion;
  }
  
  /**
   * Get the maximum version of the extension this dependency is for.
   * 
   * @return the maximum version. May be null if there is no maximum required
   *    version.
   */
  public Version getMaximumVersion()
  {
    return _maxVersion;
  }
  
  /**
   * Get a string representation of this dependency.
   * 
   * @return a string representation of this dependency.
   */
  public String toString()
  {
    return "ExtensionDependency[ id="+_id+
      ", _minVersion="+String.valueOf( _minVersion )+
      ", _maxVersion="+String.valueOf( _maxVersion )+
      " ]";
  }
  
  public int hashCode()
  {
    int result = 42;
    result = 37 * result + _id.hashCode();
    if ( _minVersion != null )
    {
      result = 37 * result + _minVersion.hashCode();
    }
    if ( _maxVersion != null )
    {
      result = 37 * result + _maxVersion.hashCode();
    }
    
    return result;
  }
  
  public boolean equals( Object o )
  {
    if ( o == this )
    {
      return true;
    }
    if ( !(o instanceof ExtensionDependency ) )
    {
      return false;
    }
    ExtensionDependency that = (ExtensionDependency) o;
    
    if ( !that.getID().equals( this.getID() ) )
    {
      return false;
    }
    
    if ( that.getMinimumVersion() != this.getMinimumVersion() )
    {
      if ( that.getMinimumVersion() != null && 
           !that.getMinimumVersion().equals( this.getMinimumVersion() ) )
      {
        return false;
      }
    }
    
    if ( that.getMaximumVersion() != this.getMaximumVersion() )
    {
      if ( that.getMaximumVersion() != null &&
           !that.getMaximumVersion().equals( this.getMaximumVersion() ) )
      {
        return false;
      }
    }
    
    return true;
  }
}
