/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.Extension;
import javax.ide.extension.ExtensionDependency;
import javax.ide.extension.ExtensionHook;
import javax.ide.util.Version;

/**
 * Visitor implementation for &lt;dependencies&gt;
 */
public class DependenciesVisitor extends ElementVisitor
{
  public final static ElementName ELEMENT = new ElementName( 
    ExtensionHook.MANIFEST_XMLNS, "dependencies" );
  private final static ElementName IMPORT = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "import" );

  private ElementVisitor _importVisitor = new ImportVisitor();
  
  private static final String MIN_VERSION = "minVersion";
  private static final String MAX_VERSION = "maxVersion";

  public final void start( ElementStartContext context )
  {
    context.registerChildVisitor( IMPORT, _importVisitor );
  }
  
  public final void end( ElementEndContext context )
  {
    endDependencies( context.getExtension() );
  }

  protected void endDependencies( Extension extension ) 
  {
    
  }

  /**
   * Add a dependency. This implementation just calls 
   * DefaultExtension.addDependency().
   * 
   * @param context the processing context.
   * @param extension the extension to add a dependency for
   * @param dep the dependency to add.
   */
  protected void addDependency( ElementContext context, 
    DefaultExtension extension, ExtensionDependency dep )
  {
    extension.addDependency( dep );    
  }
  
  private class ImportVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      String minVersion = context.getAttributeValue( "minVersion" );
      String maxVersion = context.getAttributeValue( "maxVersion" );
      
      Version vMinVersion = null;
      if ( minVersion != null )
      {
        vMinVersion = new Version( minVersion );
      }
      
      Version vMaxVersion = null;
      if ( maxVersion != null )
      {
        vMaxVersion = new Version( maxVersion );
      }
      
      context.getScopeData().put( MIN_VERSION, vMinVersion );
      context.getScopeData().put( MAX_VERSION, vMaxVersion );
    }
    
    public void end( ElementEndContext context )
    {
      String id = context.getText().trim();
      if ( id.length() == 0 )
      {
        log( context, Level.SEVERE,
          "Element 'import' must specify an extension id."
        );
        return;
      }
      ExtensionDependency dep = new ExtensionDependency( id, 
        (Version) context.getScopeData().get( MIN_VERSION ), 
        (Version) context.getScopeData().get( MAX_VERSION )
      );
      
      DefaultExtension ext = (DefaultExtension) context.getScopeData().get(
        ExtensionHook.KEY_EXTENSION
      );
      addDependency( context, ext, dep );
    }
  }
}
