/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.model;

import java.net.URI;

import java.util.prefs.Preferences;

import javax.ide.model.java.JavaModel;
import javax.ide.model.spi.ProjectImpl;
import javax.ide.net.URIPath;


/**
 * Project interface. Projects are documents encapsulating the user work.
 * A project specifies the source path where the java source files are
 * rooted, the class path used when compiling and running the project,
 * the ouput path where compiled classes are saved.
 */
public final class Project extends Document
{
  private ProjectImpl getProjectImpl()
  {
    return (ProjectImpl)getElementImpl();
  }

  /**
   * Constant identifying the project document type.
   */
  public static final String PROJECT_ID = "javax.ide.model.PROJECT_ID";

  /**
   * Identifies the bound property 'sourcePath'.
   */
  public static final String PROP_SOURCE_PATH = "sourcePath";

  /**
   * Identifies the bound property 'classPath'.
   */
  public static final String PROP_CLASS_PATH = "classPath";

  /**
   * Identifies the bound property 'outputPath'.
   */
  public static final String PROP_OUTPUT_PATH = "outputPath";

  /**
   * Get the source path. The source path where the contents of this 
   * project can be found.
   *
   * @return A {@link URIPath} pointing to where the contents of this project
   * can be located.
   */
  public URIPath getSourcePath()
  {
    return getProjectImpl().getSourcePath();
  }

  /**
   * Get the class path. The class path points to where the compiled classes
   * used by this project can be found.
   *
   * @return A {@link URIPath} pointing to where compiled classes used by this 
   * project are located.
   */
  public URIPath getClassPath()
  {
    return getProjectImpl().getClassPath();
  }

  public void setOutputDirectory(URI outputDirectory)
  {
    getProjectImpl().setOutputDirectory( outputDirectory );
  }

  public Preferences getPreferences()
  {
    return getProjectImpl().getPreferences();
  }

  /**
   * Get the {@link JavaModel} associated with this project.
   *
   * @return The {@link JavaModel} associated with this project.
   */
  public JavaModel getJavaModel()
  {
    return getProjectImpl().getJavaModel();
  }

  public void addClassPath( URIPath path )
  {
    getProjectImpl().addClassPath( path );
  }

  /**
   * Get the {@link URI} where the output from compiling project sources is
   * located.
   *
   * @return The {@link URI} where the project class files are located.
   */
  public URI getOutputDirectory()
  {
    return getProjectImpl().getOutputDirectory();
  }
}
