/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.net.spi;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.MetaClassVisitor;
import javax.ide.util.MetaClass;

/**
 * The virtual file system hook.
 */
public final class VFSHook extends ExtensionHook
{
  public static final ElementName ELEMENT = new ElementName( 
    ExtensionHook.MANIFEST_XMLNS, "vfs-hook"
  );
  
  private static final ElementName SCHEME = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "scheme"
  );
  private static final ElementName HELPER = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "helper"
  );
  private static final ElementName HELPER_CLASS = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "helper-class"
  );
  private static final String CURRENT_HOLDER = "currentHolder";
  
  private final ElementVisitor _helperVisitor = new HelperVisitor();
  private final ElementVisitor _schemeVisitor = new SchemeVisitor();
  private final ElementVisitor _helperClassVisitor = new HelperClassVisitor();
  
  private Map _helperMetaClassesByScheme = new HashMap();
  
  public void start( ElementStartContext context )
  {
    context.registerChildVisitor( HELPER, _helperVisitor );
  }
    
  private final class HelperVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      context.getScopeData().put( CURRENT_HOLDER, new Holder() );
      
      context.registerChildVisitor( SCHEME, _schemeVisitor );
      context.registerChildVisitor( HELPER_CLASS, _helperClassVisitor );
    }
    
    public void end( ElementEndContext context )
    {
      Holder holder = (Holder) context.getScopeData().get( CURRENT_HOLDER );
      
      if ( holder.scheme == null )
      {
        log(context, Level.SEVERE, "Missing required element 'scheme'.");
        return;
      }
      if ( holder.metaClass == null )
      {
        log( context, Level.SEVERE, "Missing required element 'helper-class'." );
        return;
      }
      
      if ( _helperMetaClassesByScheme.containsKey( holder.scheme ) )
      {
        log( context, Level.WARNING, 
          "Multiple virtual file system helpers for scheme '"+holder.scheme+"'. "+
          holder.metaClass.getClassName()+" takes precedence." );
      }
      
      _helperMetaClassesByScheme.put( holder.scheme, holder.metaClass );
    }
    
  }

  private final class SchemeVisitor extends ElementVisitor
  {
    public void end( ElementEndContext context )
    {
      if ( context.getText() != null )
      {
        String scheme = context.getText().trim();
        Holder holder = (Holder) context.getScopeData().get( CURRENT_HOLDER );
        holder.scheme = scheme;
      }
    }
  }
  
  private final class HelperClassVisitor extends MetaClassVisitor
  {
    protected void metaClass( ElementContext context, MetaClass clz )
    {
      Holder holder = (Holder) context.getScopeData().get( CURRENT_HOLDER );
      holder.metaClass = clz;
    }
  }
  
  
  private final class Holder
  {
    private String scheme;
    private MetaClass metaClass;
  }
}
