/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.property.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.I18NStringVisitor;
import javax.ide.property.PropertyPageRegistry;
import javax.ide.util.MetaClass;

/**
 * IDE property pages information gathered from processing the
 * <b>property-hook</b> section of an extension manifest.
 * The information recorded here describes property pages which are
 * generally given to the {@link javax.ide.property.PropertyPageRegistry} for
 * registration.<p>
 */
public final class PropertyHook extends ExtensionHook
{
  public static final ElementName ELEMENT = new ElementName( 
    ExtensionHook.MANIFEST_XMLNS, "property-hook"
  );
  
  private static final ElementName PROPERTY_PAGES = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "property-pages" );
  private static final ElementName PROPERTY_PAGE = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "property-page" );
  private static final ElementName LABEL = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "label" );
  private static final ElementName OBJECT_CLASS = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "object-class" );
    
  private static final String KEY_INFO = "propertyInfo";


  private final ElementVisitor _propertyPagesVisitor = new PropertyPagesVisitor();
  private final ElementVisitor _propertyPageVisitor = new PropertyPageVisitor();
  private final ElementVisitor _labelVisitor = new LabelVisitor();
  private final ElementVisitor _objectClassVisitor = new ObjectClassVisitor();
  
  private Collection _propertyPageInfos = new ArrayList();

  public Collection getPropertyPageInfos()
  {
    return _propertyPageInfos;
  }

  public void start( ElementStartContext context )
  {
    context.registerChildVisitor( PROPERTY_PAGES, _propertyPagesVisitor );
  }
  
  private class PropertyPagesVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      context.registerChildVisitor( PROPERTY_PAGE, _propertyPageVisitor );
    }
  }
  
  private final class PropertyPageVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      String clz = context.getAttributeValue( "property-page-class" );
      if (clz != null) clz = clz.trim();
      if ( clz == null || clz.length() == 0 )
      {
        log( context, Level.SEVERE, 
          "Missing required attribute 'property-page-class'" );
        return;
      }
      // bug 8464097
      ClassLoader cl = ElementVisitor.getClassLoader(context);
      
//      ClassLoader cl = (ClassLoader)context.getScopeData().get( 
//        ExtensionVisitor.KEY_CLASSLOADER
//      );
      MetaClass pageClass = new MetaClass( cl, clz );
      
      PropertyPageInfo info = new PropertyPageInfo();
      info.setPageClass( pageClass );
      
      String parent = context.getAttributeValue( "parent-page" );
      info.setParentPage( parent );
      
      context.getScopeData().put( KEY_INFO, info );
      
      context.registerChildVisitor( LABEL, _labelVisitor );
      context.registerChildVisitor( OBJECT_CLASS, _objectClassVisitor );
    }
    
    public void end( ElementEndContext context )
    {
      PropertyPageInfo info = getInfo( context );
      if ( info.getLabel() == null )
      {
        log( context, Level.SEVERE, "Label required.");
        return;
      }
      if ( info.getObjectClass() == null )
      {
        log( context, Level.SEVERE, "Object class required.");
        return;
      }
      _propertyPageInfos.add( info );
    }
  }
  
  private final class LabelVisitor extends I18NStringVisitor
  {
    public void string( ElementContext context, String value )
    {
      PropertyPageInfo info = getInfo( context );
      info.setLabel( value );
    }
  }
  
  private final class ObjectClassVisitor extends ElementVisitor
  {
    public void end( ElementEndContext context )
    {
      String text = context.getText();
      if (text != null) text = text.trim();
      if ( text != null && text.length() > 0 )
      {
        if ( !PropertyPageRegistry.IDE_CLASS_NAME.equals( text ) &&
             !PropertyPageRegistry.PROJECT_CLASS_NAME.equals( text ) )
        {
          log( context, Level.SEVERE, 
            "Must be either '"+PropertyPageRegistry.IDE_CLASS_NAME+"' or '"+
            PropertyPageRegistry.PROJECT_CLASS_NAME+"'." );
          return;
        }
        getInfo( context ).setObjectClass( text );
      }
    }
  }
  
  private PropertyPageInfo getInfo( ElementContext context )
  {
    return (PropertyPageInfo) context.getScopeData().get( KEY_INFO );
  }
  
  
}
