/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.util;


/**
 * Class used to identify objects. A <code>ID</code> is made up of the
 * object type and a object name. In general, the object type is a generic
 * type identifying similar objects and the object name is the specific
 * name of the object. By convention, the object type identifier is a
 * dash or dot separated string whose uniqueness comes from following
 * the package naming scheme of an extension.<p>
 *
 * In general, an IDE has a set of objects that extensions need to get a
 * hold of. Such objects, which include actions and views for example, 
 * implment the {@link javax.ide.Identifiable} interface and have an
 * <code>ID</code> that uniquely identifies them.
 */
public final class ID 
{
  private String _type;
  private String _name;

  /**
   * Constructor.
   *
   * @param type the object type. In general, the object type is a generic
   * type identifying a class of objects. The type may be null.<p>
   *
   * By convention, the object type identifier is a dash or dot separated 
   * string whose uniqueness comes from following the package naming scheme 
   * of an extension.
   *
   * @param name the object name. In general, the object name is specific
   * to the object instance. The name cannot be null.
   */
  public ID( String type, String name )
  {
    if ( type == null )
    {
      type = "";
    }

    _type = type;

    _name = name;
  }

  /**
   * Construct an <code>ID</code> with the specified name. The id type is
   * undefined.
   */
  public ID( String name )
  {
    this( null, name );
  }

  /**
   * Get the object type identifier. The type may <code>null</code>.<p>
   *
   * By convention, the object type identifier is a dash or dot separated 
   * string whose uniqueness comes from following the package naming scheme 
   * of an extension.
   * @return the type part of the object id.
   */
  public String getType()
  {
    return _type;
  }

  /**
   * Get the object name. In general, the object name is what identifies
   * instances with the same type identifier. The name cannot be 
   * <code>null</code>.
   * @return the name part of the object id. 
   */
  public String getName()
  {
    return _name;
  }

  //--------------------------------------------------------------------------
  // Object overrides...
  //--------------------------------------------------------------------------

  public String toString()
  {
    final String type = getType();

    return type.length() != 0 
           ? type + "-:-" + getName()
           : getName();
  }

  public boolean equals( Object object ) 
  {
    if ( object == this )
    {
      return true;
    }

    if ( ! (object instanceof ID ) )
    {
      return false;
    }
    final ID id = (ID) object;

    return _type.equals( id._type ) && _name.equals( id._name );
  }

  public int hashCode()
  {
    return toString().hashCode();
  }
}
