CREATE OR REPLACE TYPE MDSYS.type_rdf_model FORCE AS OBJECT(
OWNER                       VARCHAR2(128), 
MODEL_ID                    NUMBER,        
MODEL_NAME                  VARCHAR2(128),  
TABLE_NAME                  VARCHAR2(128), 
COLUMN_NAME                 VARCHAR2(128),  
MODEL_TABLESPACE_NAME       VARCHAR2(128),  
MODEL_TYPE                  VARCHAR2(40),
INMEMORY                    VARCHAR2(1)
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_model FORCE AS TABLE OF type_rdf_model;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfModels(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_model PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);  
prefix         VARCHAR2(70); 
model_cv       sys_refcursor;
OWNER                       VARCHAR2(128); 
MODEL_ID                    NUMBER;        
MODEL_NAME                  VARCHAR2(128);  
TABLE_NAME                  VARCHAR2(128); 
COLUMN_NAME                 VARCHAR2(128);  
MODEL_TABLESPACE_NAME       VARCHAR2(128);  
MODEL_TYPE                  VARCHAR2(40);
INMEMORY                    VARCHAR2(1);
BEGIN
  prefix := 'MDSYS.'; 
  IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN 
    prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
  END IF; 
  selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'rdf_model$');
  open model_cv for selstmt;
  LOOP
    fetch model_cv into OWNER, MODEL_ID,MODEL_NAME,TABLE_NAME,COLUMN_NAME,MODEL_TABLESPACE_NAME,MODEL_TYPE,INMEMORY;
    exit when (model_cv%notfound);
    PIPE ROW (type_rdf_model(OWNER, MODEL_ID,MODEL_NAME,TABLE_NAME,COLUMN_NAME,MODEL_TABLESPACE_NAME,MODEL_TYPE,INMEMORY));
  END LOOP;
END;
/

grant execute on MDSYS.getRdfModels to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_rulebase FORCE AS OBJECT(
OWNER              VARCHAR2(128),
RULEBASE_NAME      VARCHAR2(128),
RULEBASE_VIEW_NAME VARCHAR2(133),
STATUS             VARCHAR2(30)
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_rulebase FORCE AS TABLE OF type_rdf_rulebase;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfRulebases(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_rulebase PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
OWNER              VARCHAR2(128);
RULEBASE_NAME      VARCHAR2(128);
RULEBASE_VIEW_NAME VARCHAR2(133);
STATUS             VARCHAR2(30);
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_rulebase_info');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into OWNER, RULEBASE_NAME,RULEBASE_VIEW_NAME,STATUS;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_rulebase(OWNER, RULEBASE_NAME,RULEBASE_VIEW_NAME,STATUS));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfRulebases to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_rules_index FORCE AS OBJECT(
OWNER           VARCHAR2(128),
INDEX_NAME      VARCHAR2(128),
INDEX_VIEW_NAME VARCHAR2(133),
STATUS          VARCHAR2(30),
MODEL_COUNT     NUMBER,
RULEBASE_COUNT  NUMBER
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_rules_index FORCE AS TABLE OF type_rdf_rules_index;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfEntailments(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_rules_index PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
OWNER           VARCHAR2(128);
INDEX_NAME      VARCHAR2(128);
INDEX_VIEW_NAME VARCHAR2(133);
STATUS          VARCHAR2(30);
MODEL_COUNT     NUMBER;
RULEBASE_COUNT  NUMBER;
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_rules_index_info');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into OWNER, INDEX_NAME,INDEX_VIEW_NAME,STATUS,MODEL_COUNT,RULEBASE_COUNT;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_rules_index(OWNER, INDEX_NAME,INDEX_VIEW_NAME,STATUS,MODEL_COUNT,RULEBASE_COUNT));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfEntailments to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_dtype_index FORCE AS OBJECT(
DATATYPE         VARCHAR2(51),
INDEX_NAME       VARCHAR2(128),
STATUS           VARCHAR2(8),
TABLESPACE_NAME  VARCHAR2(30),
FUNCIDX_STATUS   VARCHAR2(8)
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_dtype_index FORCE AS TABLE OF type_rdf_dtype_index;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfDatatypes(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_dtype_index PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
DATATYPE         VARCHAR2(51);
INDEX_NAME       VARCHAR2(128);
STATUS           VARCHAR2(8);
TABLESPACE_NAME  VARCHAR2(30);
FUNCIDX_STATUS   VARCHAR2(8);
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_dtype_index_info');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into DATATYPE, INDEX_NAME,STATUS,TABLESPACE_NAME,FUNCIDX_STATUS;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_dtype_index(DATATYPE,INDEX_NAME,STATUS,TABLESPACE_NAME,FUNCIDX_STATUS));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfDatatypes to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_network_index FORCE AS OBJECT(
NAME                      VARCHAR2(128),
TYPE                      VARCHAR2(10),
ID                        NUMBER,
INDEX_CODE                VARCHAR2(100),
INDEX_NAME                VARCHAR2(128),
LAST_REFRESH              TIMESTAMP(6) WITH TIME ZONE,
STATUS                    VARCHAR2(8),
TABLESPACE_NAME           VARCHAR2(30),
UNIQUENESS                VARCHAR2(9),
COMPRESSION               VARCHAR2(13),
PREFIX_LENGTH             NUMBER,
BLEVEL                    NUMBER,
LEAF_BLOCKS               NUMBER,
NUM_ROWS                  NUMBER,
DISTINCT_KEYS             NUMBER,
AVG_LEAF_BLOCKS_PER_KEY   NUMBER,
AVG_DATA_BLOCKS_PER_KEY   NUMBER,
CLUSTERING_FACTOR         NUMBER,
SAMPLE_SIZE               NUMBER,
LAST_ANALYZED             DATE
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_network_index FORCE AS TABLE OF type_rdf_network_index;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfNetworkIndexes(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_network_index PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
NAME                      VARCHAR2(128);
IDX_TYPE                  VARCHAR2(10);
ID                        NUMBER;
INDEX_CODE                VARCHAR2(100);
INDEX_NAME                VARCHAR2(128);
LAST_REFRESH              TIMESTAMP(6) WITH TIME ZONE;
STATUS                    VARCHAR2(8);
TABLESPACE_NAME           VARCHAR2(30);
UNIQUENESS                VARCHAR2(9);
COMPRESSION               VARCHAR2(13);
PREFIX_LENGTH             NUMBER;
BLEVEL                    NUMBER;
LEAF_BLOCKS               NUMBER;
NUM_ROWS                  NUMBER;
DISTINCT_KEYS             NUMBER;
AVG_LEAF_BLOCKS_PER_KEY   NUMBER;
AVG_DATA_BLOCKS_PER_KEY   NUMBER;
CLUSTERING_FACTOR         NUMBER;
SAMPLE_SIZE               NUMBER;
LAST_ANALYZED             DATE;
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_network_index_info');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into NAME,IDX_TYPE,ID,INDEX_CODE,INDEX_NAME,
                            LAST_REFRESH,STATUS,TABLESPACE_NAME,UNIQUENESS,
                            COMPRESSION,PREFIX_LENGTH,BLEVEL,LEAF_BLOCKS,
                            NUM_ROWS,DISTINCT_KEYS,AVG_LEAF_BLOCKS_PER_KEY,AVG_DATA_BLOCKS_PER_KEY,
                            CLUSTERING_FACTOR,SAMPLE_SIZE,LAST_ANALYZED;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_network_index(NAME,IDX_TYPE,ID,INDEX_CODE,INDEX_NAME,
                          LAST_REFRESH,STATUS,TABLESPACE_NAME,UNIQUENESS,
                          COMPRESSION,PREFIX_LENGTH,BLEVEL,LEAF_BLOCKS,
                          NUM_ROWS,DISTINCT_KEYS,AVG_LEAF_BLOCKS_PER_KEY,AVG_DATA_BLOCKS_PER_KEY,
                          CLUSTERING_FACTOR,SAMPLE_SIZE,LAST_ANALYZED));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfNetworkIndexes to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_virtual_model_info FORCE AS OBJECT(
OWNER               VARCHAR2(128),
VIRTUAL_MODEL_NAME  VARCHAR2(128),
UNIQUE_VIEW_NAME    VARCHAR2(128),
DUPLICATE_VIEW_NAME VARCHAR2(128),
STATUS              VARCHAR2(30),
MODEL_COUNT         NUMBER,
RULEBASE_COUNT      NUMBER,
RULES_INDEX_COUNT   NUMBER
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_virtual_model_info FORCE AS TABLE OF type_rdf_virtual_model_info;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfVirtualModelInfo(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_virtual_model_info PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
OWNER               VARCHAR2(128);
VIRTUAL_MODEL_NAME  VARCHAR2(128);
UNIQUE_VIEW_NAME    VARCHAR2(128);
DUPLICATE_VIEW_NAME VARCHAR2(128);
STATUS              VARCHAR2(30);
MODEL_COUNT         NUMBER;
RULEBASE_COUNT      NUMBER;
RULES_INDEX_COUNT   NUMBER;
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_vmodel_info');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into OWNER,VIRTUAL_MODEL_NAME,UNIQUE_VIEW_NAME,
                            DUPLICATE_VIEW_NAME,STATUS,MODEL_COUNT,RULEBASE_COUNT,RULES_INDEX_COUNT;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_virtual_model_info(OWNER,VIRTUAL_MODEL_NAME,UNIQUE_VIEW_NAME,
                    DUPLICATE_VIEW_NAME,STATUS,MODEL_COUNT,RULEBASE_COUNT,RULES_INDEX_COUNT));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfVirtualModelInfo to public;

CREATE OR REPLACE TYPE MDSYS.type_rdf_virtual_model_datasets FORCE AS OBJECT(
VIRTUAL_MODEL_NAME       VARCHAR2(128),
DATA_TYPE                VARCHAR2(8),
DATA_NAME                VARCHAR2(128)
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_virtual_model_datasets FORCE AS TABLE OF type_rdf_virtual_model_datasets;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfVirtualModelDatasets(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_virtual_model_datasets PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
VIRTUAL_MODEL_NAME       VARCHAR2(128);
DATA_TYPE                VARCHAR2(8);
DATA_NAME                VARCHAR2(128);
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_vmodel_datasets');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into VIRTUAL_MODEL_NAME,DATA_TYPE,DATA_NAME;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_virtual_model_datasets(VIRTUAL_MODEL_NAME,DATA_TYPE,DATA_NAME));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfVirtualModelDatasets to public;


CREATE OR REPLACE TYPE MDSYS.type_rdf_index_datasets FORCE AS OBJECT(
INDEX_NAME       VARCHAR2(128),
DATA_TYPE        VARCHAR2(8),
DATA_NAME        VARCHAR2(128)
);
/

CREATE OR REPLACE TYPE MDSYS.type_table_rdf_index_datasets FORCE AS TABLE OF type_rdf_index_datasets;
/

CREATE OR REPLACE FUNCTION MDSYS.getRdfIndexDatasets(networkOwner VARCHAR2 default null, networkName VARCHAR2 default null)
RETURN type_table_rdf_index_datasets PIPELINED  authid current_user
AS
selstmt        VARCHAR2(300);
prefix         VARCHAR2(70);
model_cv       sys_refcursor;
INDEX_NAME       VARCHAR2(128);
DATA_TYPE        VARCHAR2(8);
DATA_NAME        VARCHAR2(128);
BEGIN
    prefix := 'MDSYS.';
    IF (networkOwner IS NOT NULL AND networkName IS NOT NULL AND networkOwner != 'MDSYS') THEN
        prefix := sys.dbms_assert.enquote_name(networkOwner, FALSE) || '.' || networkName || '#';
    END IF;
    selstmt := 'select * from ' || sys.dbms_assert.qualified_sql_name(prefix || 'sem_rules_index_datasets');

    open model_cv for selstmt;
    LOOP
        fetch model_cv into INDEX_NAME,DATA_TYPE,DATA_NAME;
        exit when (model_cv%notfound);
        PIPE ROW (type_rdf_index_datasets(INDEX_NAME,DATA_TYPE,DATA_NAME));
    END LOOP;
END;
/

grant execute on MDSYS.getRdfIndexDatasets to public;


