#!/usr/local/bin/perl -w
use strict;
use Getopt::Long;
use JSON::PP;
use Data::Dumper;
use Config;
use File::Spec;
use IPC::Open3;
use IO::Select;

sub createDB();
sub createDBFromDBDef();
sub destroyDB();
sub startDB();
sub stopDB();
sub disconnectUnloadDB();
sub dbUnload();
sub disconnectDB();
sub checkDisconnectStatus();
sub waitForDb($$);
sub sshCommand();
sub createGrid();
sub addNewHosts();
sub createHost($);
sub createInstallation($$$);
sub createInstance($$);
sub checkInstanceLocation();
sub applyModel();
sub printOSNameShell();
sub printSystemInfo();
sub deleteFiles();
sub printConnectionInfo($$$$$);
sub printConnectionCount();
sub printConnectionDetails();
sub checkActMgmtInst();
sub invalidJsonVer($);
sub trim;

my $SUCCESS = 0;
my $WAIT_FOR_DB_TIMEOUT = 3;
my $SSH_TIMEOUT = 4;
# Bug 27026929: Use a long default timeout for modelApply command (2 hours = 7,200 secs).
# This will allow time consuming operations to be completed.
my $DEFAULT_MODELAPPLY_TIMEOUT = 7200;
my $CLOSED_DB_CODE = 24;

my $command   = "";
my $connName;
my $dbDefPath = "";
my $dbName;
my $destroyDBDef=0;
my $cmdOutput;
my $gridFile = "";
my $hostFile = "";
my @delFiles;
my $instanceName;
my $timeout;
my $externalAddr = "";
my $home = "";
my $sshCommand = "";
my $internalAddress = "";
my $instanceLocation="";
my $dbDisconnect;
my $forceDisconnect;


{
  package Timer;

  use Class::Struct;
  use Time::HiRes;
  use Time::HiRes qw(gettimeofday tv_interval);

  struct ('Timer', { startTime => '$', endTime => '$'});

  sub start() {
    my $self = shift;
    $self->{'Timer::startTime'} = [gettimeofday];
  }

  sub stop() {
    my $self = shift;
    $self->{'Timer::endTime'} = [gettimeofday];
  }

  sub getElapsedTime() {
    my $self = shift;
    return tv_interval( $self->{'Timer::startTime'}, $self->{'Timer::endTime'});
  }
}

#All options are defined here. However,not all are defined.
GetOptions(   "command=s"      => \$command,
              "connName=s"     => \$connName,
              "dbDefPath=s"    => \$dbDefPath,
              "dbName=s"       => \$dbName,
              "destroyDBDef=i" => \$destroyDBDef,
              "gridFile=s"     => \$gridFile,
              "hostFile=s"     => \$hostFile,
              "file=s"         => \@delFiles,
              "instance=s"     => \$instanceName,
              "timeout=i"      => \$timeout,
              "externalAddr=s" => \$externalAddr,
              "home=s"         => \$home,
              "sshCommand=s"   => \$sshCommand,
              "internalAddr=s" => \$internalAddress,
              "instanceLocation=s" => \$instanceLocation,
              "dbDisconnect"   => \$dbDisconnect,
              "forceDisconnect" => \$forceDisconnect
);

if($command eq "createDB"){
  if($dbDefPath =~ /.*\/(.+).dbdef$/){
    $dbName = $1;
  }else{
    print "Incorrect dbdef file path";
    exit 1;
  }
  exit createDB();
}elsif($command eq "createDBFromDBDef"){
  exit createDBFromDBDef();
}elsif( $command eq "destroyDB"){
  exit destroyDB();
}elsif( $command eq "startDB"){
  exit startDB();
}elsif( $command eq "stopDB"){
  exit stopDB();
}elsif( $command eq "disconnectUnloadDB"){
  exit disconnectUnloadDB();
}elsif( $command eq "createGrid"){
  exit createGrid();
}elsif( $command eq "addNewHosts"){
  exit addNewHosts();
}elsif( $command eq "applyModel"){
  exit applyModel();
}elsif( $command eq "printOSNameShell"){
  exit printOSNameShell();
}elsif( $command eq "printSystemInfo"){
  exit printSystemInfo();
}elsif( $command eq "deleteFiles"){
  exit deleteFiles();
}elsif( $command eq "printConnectionCount"){
  exit printConnectionCount();
}elsif( $command eq "printConnectionDetails"){
  exit printConnectionDetails();
}elsif( $command eq "checkActMgmtInst"){
  exit checkActMgmtInst();
}elsif ($command eq "sshCommand"){
  exit sshCommand();
}elsif ($command eq "checkInstanceLocation"){
  exit checkInstanceLocation();
}else{
  print "Command not recognized\n";
  exit 1;
}

#Closes, unloads and destroys a database. If specified, the definition is also destroyed.
#Subsequent steps are performed even when previous ones fail.
sub destroyDB(){
  #my ($dbName,$destroyDBDef) = @_;
  #my $cmdOutput;

  print "Closing database...\n";
  $cmdOutput = `ttGridAdmin -dbClose $dbName 2>&1`;
  if($? != $SUCCESS){
    print " $cmdOutput";
  }else{
    waitForDb('closed',$dbName);
    #Closed should be fetched in the overall state?
  }

  print "Unloading database...\n";
  $cmdOutput = `ttGridAdmin -dbUnload $dbName 2>&1`;
  if($? != $SUCCESS){
    print " $cmdOutput";
  }else{
    waitForDb('unloaded',$dbName);
  }

  print "Destroying database...\n";
  $cmdOutput = `ttGridAdmin -dbDestroy $dbName 2>&1`;
  if($? != $SUCCESS){
    print " $cmdOutput";
    if($destroyDBDef != 1){
      return 1; #If we don't need to destroy the dbDef and dbDestroy didn't succeed, we failed.
    }
  }else{
    waitForDb('destroyed',$dbName);
  }
  if($destroyDBDef == 1){
    print "Destroying  database definition...\n";
    $cmdOutput = `ttGridAdmin -dbDefDelete $dbName -cascade 2>&1`;
    if($? != $SUCCESS){
      print " $cmdOutput";
      return 1;
    }
  }
  return $SUCCESS;
}

#Creates a database from a dbdef that already exists in the model. If the database exists already, dbCreate is not issued.
sub createDBFromDBDef(){

  print "Verifying if database exists...\n";
  $cmdOutput = `ttGridAdmin -dbStatus $dbName 2>&1`;
  if( $? == $SUCCESS){  #If the command succeeds, the database is created already.
    print "Database $dbName exists.";
    return 1; #Return a flag different from success to indicate that this DB exists already.
  }else{
  print "Creating database...\n";
    $cmdOutput = `ttGridAdmin -dbCreate $dbName 2>&1`;
    if( $? != $SUCCESS){
      print " $cmdOutput";
      return $?;
    }
  }

  return $SUCCESS;
}

#Creates a database from a dbdef file. Optionally, open the database after creation.
sub createDB(){
  #my ($dbDefPath,$dbName,$openDB) = @_;
  #my $cmdOutput;
  print "Creating Database Definition...\n";
  $cmdOutput = `ttGridAdmin -dbDefCreate $dbDefPath 2>&1`;
  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }

  $cmdOutput = applyModel();
  if($cmdOutput != $SUCCESS){
    return 1;
  }

  print "Creating database...\n";
  $cmdOutput = `ttGridAdmin -dbCreate $dbName 2>&1`;
  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }

  #While db is not open do not follow
  waitForDb('loaded',$dbName);

  #Distribution on instances happens when the dbdef file has two comments in the first line
  #The first one serves as a flag that db distribute should take place
  #The second one has the name of the instances to include in the distribution map separated by spaces
  open(FILE, $dbDefPath);
  my $firstLine = <FILE>;
  my $secondLine = <FILE>;

  chomp $firstLine;             #We remove the new line character
  if($firstLine eq "#!!!!!SQLDeveloper:DBDistribution!!!!!"){       #And see if the first line is this comment
    my $instancesLine = substr $secondLine,1;       #Next, we read the second line which is a comment and remove the # sign
    my @instances = split / /, $instancesLine;      #Split the instances names
    my $addInstCmd = "";

    foreach my $inst (@instances){      #And generate the command for adding to the distribution map
       $addInstCmd = $addInstCmd." -add $inst";
    }
    chomp $addInstCmd;          #Remove new line character
    print "Distributing database on selected instances...\n" ;
    $cmdOutput = `ttGridAdmin -dbDistribute $dbName $addInstCmd -apply 2>&1`;
      if( $? != $SUCCESS){
        print " $cmdOutput";
        return 1;
      }
    }

  close FILE;

  $cmdOutput = `rm -f $dbDefPath 2>&1`;

  return $SUCCESS;
}
#Send an ssh command to data instances using ssh from the mmgmt to the internal address of data instances.
sub sshCommand(){
  system "ssh -x $internalAddress $sshCommand";
  return $? >> 8;
}
# Waits for a status of a database. Issues ttGridAdmin -dbStatus $dbName until the database or its elements are in a certain state
# This function is a slight modification of the setupGrid script function to wait for dbStatus.
sub waitForDb($$){
  my ($state,$dbname) = @_;
  my $maxTries = 60;  #We will try to retrieve dbStatus $maxTries times...
  my $counter = 0;
  my $interval = 5; #...at an interval of $interval seconds
  my $dbinfo;
  my $jsonOutput;
  my $condition;
  my $printCond;

  # what are we waiting for? create the waiting condition...
  if($state =~ /^create(d)?/){
    $condition = q/($nInstances == $nInstancesCreated)/;
    $printCond = q/ print "$nInstancesCreated created out of $nInstances"/;
  }
  elsif($state =~ /^load(ed)?/){
    $condition = q/($nInstances == $nInstancesLoaded)/;
    $printCond = q/ print "$nInstancesLoaded loaded out of $nInstances"/;
  }
  elsif($state =~ /^open(ed)?/){
    $condition = q/($dbinfo->{opened} eq 'Y' || $nInstances == $nInstancesOpened)/;
    $printCond = q/ print "$nInstancesOpened opened out of $nInstances"/;
  }
  elsif($state =~ /^close(d)?/){
    $condition = q/(defined($dbinfo) and ($dbinfo->{opened} eq 'N'))/;
    $printCond = q/print ""/;
  }
  elsif($state =~ /^unload(ed)?/){
    $condition = q/(defined($dbinfo) and ($nInstances == $nInstancesUnLoaded))/;
    $printCond = q/print "$nInstancesUnLoaded unloaded out of $nInstances"/;
  }
  elsif($state =~ /^destroy(ed)?/){
    $condition = q/(defined($dbinfo)/;
    $printCond = q/print "$nInstancesDestroyed destroyed out of $nInstances"/;
  }
  elsif($state =~ /^partial(y)?/){
    $condition = q/($nInstancesCreated < 1)/;
    $printCond = q/print ""/;
  }
  print "Waiting for state '$state' \n";

  my $cond = 1;
  do{
    my $nInstances = 0;
    my $nInstancesOpened = 0;
    my $nInstancesLoaded = 0;
    my $nInstancesCreated = 0;
    my $nInstancesUnLoaded = 0;
    my $nInstancesDestroyed = 0;
    sleep $interval;
    $cmdOutput = `ttGridAdmin -dbStatus $dbname -f json 2>&1`;
    #print cmdOutput;
    $jsonOutput = decode_json($cmdOutput);
    if (invalidJsonVer($jsonOutput)) {
      print "Unsupported 'jsonVer' value in ttGridAdmin's output.";
      exit 2;
    }
    #print Dumper($jsonOutput);
    $dbinfo = $jsonOutput->{databases}[0];
    #print Dumper($dbinfo);
      if(defined($dbinfo)){
        foreach my $instance (@{$dbinfo->{instances}}){
          $nInstancesOpened++ if ($instance->{opened} eq 'Y');
          foreach my $element (@{$instance->{elements}}){
       if ($element->{state} eq 'CreateFailed'){
            print "Creation failed with message $element->{lastChangeMsg}\n";
              exit 2; #Not success
            }
      #Apparently, only this four states should be read at element level.
            $nInstances++;
            $nInstancesCreated++ if ($element->{state} eq 'Loaded');
            $nInstancesLoaded++ if ($element->{state} eq 'Loaded');
            $nInstancesUnLoaded++ if ($element->{unloaded} eq 'Y');
            $nInstancesDestroyed++ if ($element->{state} eq 'Destroyed');

          }
        }
      }else{
  if($state =~ /^destroy(ed)?/){
          return $SUCCESS;
  }else{
          print "Failed to retrieve status: $cmdOutput";
    exit 2;
  }
      }
      $cond = eval $condition;
      #print "Retry condition:$condition**Eval = $cond\n".
      #      "nInstances=$nInstances\n".
      #      "nInstancesOpened=$nInstancesOpened\n".
      #      "nInstancesLoaded=$nInstancesLoaded\n".
      #      "nInstancesCreated=$nInstancesCreated\n".
      #      "nInstancesUnLoaded=$nInstancesUnLoaded\n".
      #      "nInstancesDestroyed=$nInstancesDestroyed\n" ;
      eval $printCond;
      print "\n";
      if($cond){
  return $SUCCESS;
      }
      $counter++;
    } while($counter < $maxTries);
    exit $WAIT_FOR_DB_TIMEOUT;
}


#Loads and opens a database.
sub startDB(){
  #my ($dbName) = @_;
  #my $cmdOutput;
  print "Loading database...\n";
  $cmdOutput = `ttGridAdmin -dbLoad $dbName 2>&1`;
  if( $? != $SUCCESS){
    print " $cmdOutput";
  }

  print "Opening database...\n";
  $cmdOutput = `ttGridAdmin -dbOpen $dbName 2>&1`;
  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }

  return $SUCCESS;
}

#Closes, disconnects and unloads a database.
sub stopDB(){
  my $jsonOutput;
  my $statusCode;
    
  #we start by closing the database either force disconnect is or not available.
  print "Closing $dbName ...\n"; 
  $cmdOutput = `ttGridAdmin -dbClose $dbName -o json 2>&1`;
  if($? != $SUCCESS){ 
    $jsonOutput = decode_json($cmdOutput);
    if (invalidJsonVer($jsonOutput)) {
      print "Unsupported 'jsonVer' value in ttGridAdmin's output.";
      exit 2;
    }
    $statusCode = $jsonOutput->{status};
    #if status code is not CLOSED_DB_CODE then we return 1 and print the error message. when stats is CLOSED_DB_CODE we proceed to disconnect.   
    if($statusCode != $CLOSED_DB_CODE) {
      print " $jsonOutput->{errmsg}";
      return 1;
    }
  }

  #If Force Disconnect feature is not available then when proced to unload normally
  if(!$dbDisconnect){
    return dbUnload();
  }else{
    return disconnectUnloadDB();
  }
}

#Disconnects and unloads database
sub disconnectUnloadDB(){
  my $status;
  $status = disconnectDB();
  if($status != $SUCCESS){
    return $status;
  }
  return dbUnload();
}

#Unloads the database, only when force disconnect is not available
sub dbUnload(){  
  print "Unloading $dbName ...\n";  
  $cmdOutput = `ttGridAdmin dbUnload $dbName 2>&1`;
  if($? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  return $SUCCESS;
}

#Disconnects database using the force disconnect feature and then unloads the database.
sub disconnectDB(){
  my $status;

  print "Disconnecting with transactional urgency level ...\n"; 
  $cmdOutput = `ttGridAdmin dbDisconnect $dbName -transactional 2>&1`;
  if($? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  
  $status = checkDisconnectStatus() eq $SUCCESS;

  if($status){ 
    return $SUCCESS;
  }
  else{
    print "WARNING: Disconnecting with transactional urgency level failed \n \n";
    print "Disconnecting with immediate urgency level ...\n"; 
    $cmdOutput = `ttGridAdmin dbDisconnect $dbName -immediate 2>&1`;
    if($? != $SUCCESS){
      print " $cmdOutput";
      return 1;
    }
  }
  $status = checkDisconnectStatus() eq $SUCCESS;

  if($status){
    return $SUCCESS;
  }
  else{
      
    if($forceDisconnect){
      print "WARNING: Disconnecting with immediate failed urgency level \n \n";
      print "WARNING: Disconnecting with abort urgency level ...\n"; 
      #we are going to unload here.
      $cmdOutput = `ttGridAdmin dbDisconnect $dbName -abort  2>&1`;
      if($? != $SUCCESS){
        print " $cmdOutput";
        return 1;
      }
      $status = checkDisconnectStatus() eq $SUCCESS;
    
      if($status){ 
        return $SUCCESS;
      }
      else{
        print "WARNING: Polling timeout of 10 seconds exceeded. \n";
        print "WARNING: The disconnect operation might still be in progress. \n";
        print "WARNING: Make sure all transactions have completed.  \n";
        
        return 1;
      }
    }
    else{
      print "WARNING: Polling timeout of 10 seconds exceeded. \n";
      print "WARNING: The disconnect operation might still be in progress. \n";
      print "WARNING: Make sure all transactions have completed.  \n";
        
      return 1;
    }
  }  
}

#Retrieves the status of the ttGridAdmin dbDisconnectStatus command and check its value, if the value of the disconnect is defined then the function returns true.
sub checkDisconnectStatus(){
  my $maxTries = 60; #Number of times we are going yo try to retrieve the disconnect state
  my $counter = 0;
  my $interval = 1;
  my $jsonOutput;
  my $state;
  
  while($counter < $maxTries ){
    #Search into ttGridAdmin dbDisconnectStatus
    $cmdOutput = `ttGridAdmin dbDisconnectStatus $dbName -o json`;
    if($? != $SUCCESS){
      print " $cmdOutput";
      return 1;
    }
    #print cmdOutput;
    $jsonOutput = decode_json($cmdOutput);
    if (invalidJsonVer($jsonOutput)) {
      print "Unsupported 'jsonVer' value in ttGridAdmin's output.\n";
      exit 2;
    }

    if(defined($jsonOutput->{disconnects})){
      my $state = $jsonOutput->{disconnects}[0]->{state};
      if($state eq "Complete") {
         print "Disconnection of $dbName succeeded\n"; 
         return $SUCCESS;
      }elsif($state eq "Failed"){
         print "WARNING: Disconnection of $dbName failed\n"; 
         return 1;   
      }
      print "Disconnection status of $dbName is $state\n";
      $counter = $counter + 1;
      sleep($interval);
    }
  }
  return 1;
}


# Creates a new TimesTen Scaleout.
sub createGrid(){
  my $cmdOutput;
  my $gridName = "";
  my $k;
  my $intAddress = "";
  my $extAddress = "";
  my $addrStr = "";
  my $hostName = "";
  my $dsg = "";
  my $n_dsg = 0;
  my $mgmtPort;
  my $memService = "";
  my $memConfig = "";
  my $timer = Timer->new();
  my $et = 0;

  local $/;
  open(FILE, $gridFile);
  my $data = <FILE>;
  close FILE;

  my $result = decode_json($data);
  $gridName = $result->{'gridName'};
  $k = $result->{'k'};
  $intAddress = $result->{'internalAddress'};
  $extAddress = $result->{'externalAddress'};
  $hostName = $result->{'hostName'};
  $mgmtPort = $result->{'mgmtPort'};
  $memService = $result->{'membershipService'};
  $memConfig = $result->{'membershipConfig'};
  $n_dsg = $result->{'dataSpaceGroup'};

  if ($n_dsg == 0) {
     $dsg = "-nodataspacegroup";
  }
  $addrStr = "-internalAddress ${intAddress} -externalAddress ${extAddress}";
  $timer->start();
  $cmdOutput = `ttGridAdmin -gridCreate -name $gridName -k $k $addrStr -host $hostName $dsg -mgmtport $mgmtPort -membership $memService -membershipConfig $memConfig 2>&1`;
  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
# By default, ttGridAdmin assigns to DSG 1 the host where the management instance is created.
# There is a special case in Simple TimesTen Scaleout Wizard. If the host that hosts (:)) the management
# instance also contains a data instance, then we need to assign such host to a different DSG
# if users indicate that in the third step of the wizard (DSG allocation).
  if ($n_dsg > 1) {
     print "Assigning host $hostName to data space group $n_dsg \n";
     $cmdOutput = `ttGridAdmin -hostModify -name $hostName -dataSpaceGroup $n_dsg 2>&1`;
  }
  $timer->stop();
  if($? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  else {
    $et = $timer->getElapsedTime();
    print "    Elapsed Time: $et seconds \n";
  }

  return $SUCCESS;
}

# Adds new hosts (along with installation and instance) to a recently created grid.
sub addNewHosts(){
  my $cmdOutput;
  my $intAddress = "";
  my $extAddress = "";
  my $hostName = "";
  my $instanceName = "";
  my $installationLocation = "";

  local $/;
  open(FILE, $hostFile);
  my $data = <FILE>;
  close FILE;

  my $result = decode_json($data);
  $intAddress = $result->{'internalAddress'};
  $extAddress = $result->{'externalAddress'};

  $hostName = $result->{'hostName'};
  $installationLocation = $result->{'installationLocation'};
  $instanceName = $result->{'instanceName'};

  my @hosts = @{$result->{'hosts'}};
  foreach my $host (@hosts) {
    if($host->{'internalAddress'} ne $intAddress ||
       $host->{'externalAddress'} ne $extAddress) {
      $cmdOutput = createHost($host);
      if ($cmdOutput != $SUCCESS) {
        return 1;
      }
      my @installs = @{$host->{'installations'}};
      foreach my $installation (@installs) {
        $cmdOutput = createInstallation($host->{'name'}, $installation->{'name'}, $installation->{'location'});
        if ($cmdOutput != $SUCCESS) {
          return 1;
        }
      }
      my @instances = @{$host->{'instances'}};
      foreach my $instance (@instances) {
        $cmdOutput = createInstance($host->{'name'}, $instance);
        if ($cmdOutput != $SUCCESS) {
          return 1;
        }
      }
    }
    else {
      if($host->{'name'} ne $hostName) {
        $cmdOutput = createHost($host);
        if ($cmdOutput != $SUCCESS) {
          return 1;
        }
      }
      my @installs = @{$host->{'installations'}};
      foreach my $installation (@installs) {
        if($installation->{'location'} ne $installationLocation ||
          ($installation->{'location'} eq $installationLocation && $host->{'name'} ne $hostName)) {
          $cmdOutput = createInstallation($host->{'name'}, $installation->{'name'}, $installation->{'location'});
          if ($cmdOutput != $SUCCESS) {
            return 1;
          }
        }
      }
      my @insts = @{$host->{'instances'}};
      foreach my $inst (@insts) {
        if($inst->{'name'} ne $instanceName ||
          ($inst->{'name'} eq $instanceName && $host->{'name'} ne $hostName)) {
          $cmdOutput = createInstance($host->{'name'}, $inst);
          if ($cmdOutput != $SUCCESS) {
            return 1;
          }
        }
      }
    }
  }

  return $SUCCESS;
}


sub createHost($) {
  my ($host) = @_;
  my $name = $host->{'name'};
  my $intAddr = $host->{'internalAddress'};
  my $extAddr = $host->{'externalAddress'};
  my $addrStr = "-internalAddress ${intAddr} -externalAddress ${extAddr}";
  my $dsg = $host->{'dataSpaceGroup'};
  my $dsgStr = "";
  my $cmdOutput;

  if($dsg <= 0) {
    $dsgStr = "-nodataspacegroup";
  }
  else {
    $dsgStr = "-dataspacegroup ${dsg}";
  }

  print "Adding host $name to grid model ...\n";

  $cmdOutput = `ttGridAdmin -hostCreate -name $name $addrStr $dsgStr`;

  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  return $SUCCESS;
}

sub createInstallation($$$) {
  my ($hName, $installName, $installLoc) = @_;
  my $cmdOutput;
  print "Adding installation $installName into host $hName to grid model...\n";

  $cmdOutput = `ttGridAdmin -installationCreate -name $installName -host $hName -location $installLoc`;

  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  return $SUCCESS;
}

sub createInstance($$) {
  my ($hName, $instance) = @_;
  my $cmdOutput;
  my $instName = $instance->{'name'};
  my $location = $instance->{'location'};
  my $installation = $instance->{'installation'};
  my $daemonPort = $instance->{'daemonPort'};
  my $csPort = $instance->{'csPort'};

  print "Adding instance $instName into host $hName to grid model...\n";

  $cmdOutput = `ttGridAdmin -instanceCreate -name $instName -host $hName -installation $installation -location $location -daemonport $daemonPort -csport $csPort`;

  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  return $SUCCESS;
}

sub checkInstanceLocation() {
    $cmdOutput = `ssh -x $internalAddress test -w $instanceLocation 2>&1`;
    if ( $? != $SUCCESS){
        if (index($cmdOutput, 'Connection timed out') != -1) {
            return $SSH_TIMEOUT;
        }
        else{
            return 1;
        }
    }
    return $SUCCESS;
}

sub applyModel() {
  my $timer = Timer->new();
  my $et = 0;

  print "Applying grid model ...\n";
  $timer->start();
  my $cmdOutput = `ttGridAdmin -modelApply -timeout $DEFAULT_MODELAPPLY_TIMEOUT 2>&1`;
  $timer->stop();
  if( $? != $SUCCESS){
    print " $cmdOutput";
    return 1;
  }
  else {
    $et = $timer->getElapsedTime();
    print "    Elapsed Time: $et seconds \n";
  }
  return $SUCCESS;
}

# Print operating system and user's shell of active management instance machine.
sub printOSNameShell() {
  print "$Config{osname};$ENV{SHELL}\n";
  return $SUCCESS;
}

# Print some OS info: OS Name, shell, temporary directory and file separator.
sub printSystemInfo() {
  my $tmpdir = File::Spec->tmpdir();
  my $fseparator = File::Spec->catfile('','');
  print "$Config{osname};$ENV{SHELL};$tmpdir;$fseparator;\n";
  return $SUCCESS;
}

# Delete a given list of files
sub deleteFiles() {
  unlink @delFiles;
  return $SUCCESS;
}
# Print json output of ttISql connection details or count
sub printConnectionInfo($$$$$){
  my ($instance_name,
      $timeout,
      $database_name,
      $connection_name,
      $details) = @_;
  my $ttIsql_cmd;

  if( $details) {
   $ttIsql_cmd = " ttisql -e 'select /*+ TT_GridQueryExec(Global) TT_PartialResult(1) */ * from sys.v\$datastore_status;exit;' -connStr 'dsn=$database_name;connectionname=$connection_name;' ";
  }
  else {
   $ttIsql_cmd = " ttisql -e 'select /*+ TT_GridQueryExec(Global) TT_PartialResult(1) */ contype,datastore, count(*) from sys.v\$datastore_status group by datastore, contype;exit;' -connStr 'dsn=$database_name;connectionname=$connection_name;' ";
  }

  my $command = 'ttGridAdmin';
  my @args = ('-instanceExec','-f','json','-only',$instance_name,'-type','data',
              '-timeout',$timeout, $ttIsql_cmd);
  return system $command, @args;
}
# Print output of ttIsql connection count
sub printConnectionCount() {
 return printConnectionInfo($instanceName, $timeout, $dbName, $connName, 0);
}
sub printConnectionDetails() {
 return printConnectionInfo($instanceName, $timeout, $dbName, $connName, 1);
}

# Check if a given instance is the active management instance of a grid.
sub checkActMgmtInst() {
  my $cmdOutput = `test -d $home`;
  if( $? != $SUCCESS){
    print "Directory '$home' does not exist in '$externalAddr'";
  return 1;
  }
  # Get instance guid
  $cmdOutput = `grep instance_guid $home/conf/timesten.conf 2>&1`;
  if($? != $SUCCESS){
    print "Error while getting instance guid: $cmdOutput";
  return 1;
  }
  $cmdOutput = trim($cmdOutput);
  my $index = index($cmdOutput, '=');
  if ($index < 0) {
    print "Cannot find instance GUID";
    return 1;
  }
  my $instGuid = substr($cmdOutput, $index + 1);

  $cmdOutput = `ttGridAdmin -mgmtStatus -f json 2>&1`;
  $cmdOutput = trim($cmdOutput);

  my $fi = index($cmdOutput, "{");
  my $ei = rindex($cmdOutput, "}");
  if ($fi == 0 && $ei == length($cmdOutput) - 1) {
    my $jsonOutput = decode_json($cmdOutput);
    if (invalidJsonVer($jsonOutput)) {
      print "Unsupported 'jsonVer' value in ttGridAdmin's output.";
      return 1;
    }
    my $one = $jsonOutput->{"1"};
    if (defined($one)) {
      my $dbActive = $one->{"dbActive"};
      if (defined($dbActive)) {
        my $guid = trim($dbActive->{"instanceGuid"});
        if (uc $guid eq uc $instGuid) {
          return 0;
        }
        else {
          print "This is not the active management instance of the grid.";
          return 1;
        }
      }
      else {
        my $errmsg = $one->{"errmsg"};
        if (defined($errmsg)) {
          print "$errmsg";
        }
        else {
          print "This is not the active management instance of the grid.";
        }
        return 1;
      }
    }
    else {
      print "Cannot find management instances in mgmStatus output.";
      return 1;
    }
  }
  else {
    print "Error while executing ttGridAdmin mgmtStatus: $cmdOutput";
    return 1;
  }

}

# Check that 'jsonVer' in ttGridAdmin output is equals to 1.
sub invalidJsonVer($) {
  my ($jsonOutput) = @_;
  my $jsonVer = $jsonOutput->{"jsonVer"};
  if (defined($jsonVer)) {
    if ($jsonVer ne "1") {
      if ($jsonVer != 1) {
        return 1;
      }
    }
  }
  return 0;
}

sub trim { my $s = shift; $s =~ s/^\s+|\s+$//g; return $s };
