/////////////////////////////////////////////////////////////
// CINEMA 4D SDK                                           //
/////////////////////////////////////////////////////////////
// (c) 1989-2004 MAXON Computer GmbH, all rights reserved  //
/////////////////////////////////////////////////////////////

#ifndef __C4DBASEOBJECT_H
#define __C4DBASEOBJECT_H

#include "c4d_baselist.h"
#include "c4d_basecontainer.h"
#include "operatingsystem.h"

class BaseTag;
class BaseDocument;
class BaseContainer;
class LineObject;
class BaseThread;
class BaseSelect;
class Neighbor;

#define MODE_ON    0
#define MODE_OFF   1
#define MODE_UNDEF 2

struct CPolygon
{
	CPolygon(void) {}
	CPolygon(LONG t_a, LONG t_b, LONG t_c) { a=t_a; b=t_b; c=d=t_c; }
	CPolygon(LONG t_a, LONG t_b, LONG t_c, LONG t_d) { a=t_a; b=t_b; c=t_c; d=t_d; }

	LONG a,b,c,d;
};

struct Tangent
{
	Vector vl,vr;
};

struct Segment
{
	LONG	cnt;
	Bool	closed;
};

struct ObjectColorProperties
{
	LONG		usecolor;
	Vector	color;
	Bool		xray;
};

class HierarchyHelp
{
	private:
		HierarchyHelp();
		~HierarchyHelp();
	public:
		Real GetLOD(void) { return C4DOS.Bd->HhGetLOD(this); }
		LONG GetVFlags(void) { return C4DOS.Bd->HhGetVFlags(this); }
		Matrix			 GetMg(void) { return C4DOS.Bd->HhGetMg(this); }
		BaseThread   *GetThread(void) { return C4DOS.Bd->HhGetThread(this); }
		BaseDocument *GetDocument(void) { return C4DOS.Bd->HhGetDocument(this); }
		void AddVFlags(LONG mask) { C4DOS.Bd->HhAddVFlags(this,mask); }
};

class BaseObject : public BaseList2D
{
	private:
		BaseObject();
		~BaseObject();
	public:
		BaseObject *GetNext    (void) { return (BaseObject*)AtCall(GetNext)(); }
		BaseObject *GetPred    (void) { return (BaseObject*)AtCall(GetPred)(); }
		BaseObject *GetUp      (void) { return (BaseObject*)AtCall(GetUp)(); }
		BaseObject *GetDown    (void) { return (BaseObject*)AtCall(GetDown)(); }
		BaseObject *GetDownLast(void) { return (BaseObject*)AtCall(GetDownLast)(); }

		Real	 GetVisibility(Real parent);

		Vector GetPos(void) { return C4DOS.Bo->GetPos(this); }
		void	 SetPos(const Vector &v) { C4DOS.Bo->SetPos(this,v); }

		Vector GetScale(void) { return C4DOS.Bo->GetScale(this); }
		void   SetScale(const Vector &v) { C4DOS.Bo->SetScale(this,v); }

		Vector GetRot(void) { return C4DOS.Bo->GetRot(this); }
		void	 SetRot(const Vector &v) { C4DOS.Bo->SetRot(this,v); }

		Matrix GetMl(void) { return C4DOS.Bo->GetMl(this); }
		void   SetMl(const Matrix &m, Bool force=TRUE) { C4DOS.Bo->SetMl(this,m,force); }

		Matrix GetMg(void) { return C4DOS.Bo->GetMg(this); }
		void   SetMg(const Matrix &m, Bool force=TRUE) { C4DOS.Bo->SetMg(this,m,force); }

		Matrix GetMln(void);
		Matrix GetMgn(void);
		Matrix GetUpMg(void) { return C4DOS.Bo->GetUpMg(this); }

		Vector GetMp (void) { return C4DOS.Bo->GetMp(this); }
		Vector GetRad(void) { return C4DOS.Bo->GetRad(this); }

		LONG	 GetEditorMode(void);
		void   SetEditorMode(LONG mode);

		LONG	 GetRenderMode(void);
		void   SetRenderMode(LONG mode);

		Bool	 GetDeformMode(void);
		void   SetDeformMode(Bool mode);

		LONG   GetUniqueIP(void);
		void	 SetUniqueIP(LONG ip);

		void				SetOrigin(BaseObject *origin);
		BaseObject* GetOrigin(Bool safe);
		BaseObject* GetTopOrigin(Bool parent, Bool safe);
		BaseObject* GetEditObject(BaseObject **psds, LONG state);

		BaseTag     *GetFirstTag(void);
		BaseTag			*GetTag(LONG type, LONG nr=0) { return C4DOS.Bo->GetTag(this,type,nr); }
		const void	*GetTagDataR(LONG type, LONG nr=0) { return C4DOS.Bo->GetTagDataR(this,type,nr); }
		void				*GetTagDataW(LONG type, LONG nr=0) { return C4DOS.Bo->GetTagData(this,type,nr); }
		LONG				GetTagDataCount(LONG type) { return C4DOS.Bo->GetTagDataCount(this,type); }
		BaseTag			*MakeTag (LONG type, BaseTag *pred=NULL);
		VariableTag *MakeVariableTag(LONG type, LONG count, BaseTag *pred=NULL);
		void				InsertTag(BaseTag *tp, BaseTag *pred=NULL);
		void        KillTag (LONG type, LONG nr=0);

		BaseObject	*GetCache(HierarchyHelp *hh=NULL) { return C4DOS.Bo->GetCache(this,hh); }
		BaseObject	*GetDeformCache(void) { return C4DOS.Bo->GetDeformCache(this); }
		LineObject	*GetIsoparm(void) { return C4DOS.Bo->GetIsoparm(this); }
		void				SetIsoparm(LineObject *l);
		BaseObject  *GetCacheParent(void);

		Bool				CopyTagsTo(BaseObject *dest, LONG visible, LONG variable, LONG hierarchical, AliasTrans *trans);

		Bool				SetPhong(Bool on, Bool anglelimit, Real angle);

		Bool				IsDirty(LONG flags)  { return C4DOS.Bo->IsDirty(this,flags); }
		void				SetDirty(LONG flags) { C4DOS.Bo->SetDirty(this,flags); }
		Bool				CheckCache(HierarchyHelp *hh) { return C4DOS.Bo->CheckCache(this,hh); }
		void				Touch(void);

		void				SetColorProperties(ObjectColorProperties *prop);
		void				GetColorProperties(ObjectColorProperties *prop);

		SplineObject *GetRealSpline(void);

		LONG				GetHighlightHandle(BaseDraw *bd) { return C4DOS.Bo->GetHighlightHandle(this,bd); }

		static BaseObject *Alloc(LONG type);
		static void Free(BaseObject *&bl);

		const Matrix &GetModelingAxis(BaseDocument *doc) { return C4DOS.Bo->GetModelingAxis(this,doc); }
		void SetModelingAxis(const Matrix &m) { C4DOS.Bo->SetModelingAxis(this,m); }

		BaseObject *GetAndCheckHierarchyClone(HierarchyHelp *hh, BaseObject *op, LONG flags, Bool *dirty, AliasTrans *trans, Bool allchilds);
		BaseObject *GetHierarchyClone(HierarchyHelp *hh, BaseObject *op, LONG flags, Bool *dirty, AliasTrans *trans);

		void NewDependenceList(void);
		Bool CompareDependenceList(void);
		void AddDependence(HierarchyHelp *hh, BaseObject *op);
		void TouchDependenceList(void);

		void RemoveFromCache(void);

		Bool SearchHierarchy(BaseObject *op) { return C4DOS.Bo->SearchHierarchy(this,op); }
};

class CameraObject: public BaseObject
{
	private:
		CameraObject();
		~CameraObject();
	public:
		LONG				GetProjection(void);
		Real				GetFocus(void);
		Real				GetZoom(void);
		Vector			GetOffset(void);
		Real				GetAperture(void);
		Bool				SetProjection(LONG projection);
		Bool				SetFocus(Real v);
		Bool				SetAperture(Real v);
		Bool				SetZoom(Real zoom);
		Bool				SetOffset(const Vector &offset);

		static CameraObject *Alloc();
		static void Free(CameraObject *&bl);
};

class PointObject: public BaseObject
{
	private:
		PointObject();
		~PointObject();
	public:
		BaseSelect	*GetPointS(void);
		BaseSelect	*GetPointH(void);

		const Vector	*GetPointR(void) { return (const Vector*)GetTagDataR(Tpoint); }
		Vector				*GetPointW(void) { return (Vector*)GetTagDataW(Tpoint); }
		LONG					GetPointCount(void) { return GetTagDataCount(Tpoint); }

		Real					*CalcVertexMap(BaseObject *modifier);

		Bool					ResizeObject(LONG pcnt);
};

struct CLine
{
	Real t,pos;
};

class LineObject : public PointObject
{
	private:
		LineObject();
		~LineObject();

	public:

		const Segment	*GetSegmentR(void)		{ return (const Segment*)GetTagDataR(Tsegment); }
		Segment				*GetSegmentW(void)	{ return (Segment*)GetTagDataW(Tsegment); }
		LONG					GetSegmentCount(void) { return GetTagDataCount(Tsegment); }

		const CLine		*GetLineR(void)  { return (const CLine*)GetTagDataR(Tline); }
		CLine					*GetLineW(void) { return (CLine*)GetTagDataW(Tline); }

		Bool					ResizeObject(LONG pcnt, LONG scnt);
		PolygonObject *Triangulate(Real regular, BaseThread *bt);

		static LineObject *Alloc(LONG pcnt, LONG scnt);
		static void Free(LineObject *&bl);
};

class PolygonObject : public PointObject
{
	private:
		PolygonObject();
		~PolygonObject();
	public:
		BaseSelect	*GetPolygonS(void);
		BaseSelect	*GetPolygonH(void);

		BaseSelect	*GetEdgeS(void);
		BaseSelect	*GetEdgeH(void);

		BaseSelect	*GetPhongBreak(void);

		const CPolygon *GetPolygonR(void)  { return (const CPolygon*)GetTagDataR(Tpolygon); }
		CPolygon			 *GetPolygonW(void) { return (CPolygon*)GetTagDataW(Tpolygon); }
		LONG				GetPolygonCount(void) { return GetTagDataCount(Tpolygon); }

		Bool				ResizeObject(LONG pcnt, LONG vcnt);
		Bool				ResizeObject(LONG pcnt, LONG vcnt, LONG ncnt);
		Bool				ResizeObject(LONG pcnt, LONG vcnt, LONG ncnt, LONG vc_flags);

		BaseSelect* GetSelectedEdges(Neighbor *e, EdgeSelectionType ltype);
		BaseSelect* GetSelectedEdges(Neighbor *e, BaseSelect* sel);
		Bool SetSelectedEdges(Neighbor *e, BaseSelect* pSel, EdgeSelectionType ltype);
		Vector *CreatePhongNormals();
		Bool GetPolygonTranslationMap(LONG &ngoncnt, LONG *&polymap); // polygon index -> ngon index
		Bool GetNGonTranslationMap(LONG ngoncnt, LONG *polymap, LONG **&ngons); // ngon index -> polygons
		Pgon *GetAndBuildNgon(void);
		LONG GetNgonCount(void);
		void GetSelectedNgons(BaseSelect* sel);
		void GetHiddenNgons(BaseSelect* sel);
		NgonBase *GetNgonBase();
		Bool ValidateEdgeSelection(BaseSelect* sel);
		Bool ValidateEdgeSelection();
		Bool GetEdgeSelection(BaseSelect* sel, EdgeSelectionType type);
		void GetNgonEdgesCompact(UCHAR *&edges);

		static PolygonObject *Alloc(LONG pcnt, LONG vcnt);
		static void Free(PolygonObject *&bl);
};

class SplineObject : public PointObject
{
	private:

		SplineObject();
		~SplineObject();

	public:

		LONG					GetInterpolationType(void);
		Bool					IsClosed(void);
		Vector				GetSplinePoint  (Real t, LONG segment=0, const Vector *padr=NULL);
		Vector				GetSplineTangent(Real t, LONG segment=0, const Vector *padr=NULL);

		Bool					InitLength(LONG segment=0, const Vector *padr=NULL);
		void					FreeLength(void);
		Real					UniformToNatural(Real t);
		Real					GetLength(void);
		Real					GetSegmentLength(LONG a, LONG b);
		Bool					SetDefaultCoeff(void);

		Bool					ResizeObject(LONG pcnt, LONG scnt);

		const Segment	*GetSegmentR(void)  { return (const Segment*)GetTagDataR(Tsegment); }
		Segment				*GetSegmentW(void) { return (Segment*)GetTagDataW(Tsegment); }
		LONG					GetSegmentCount(void) { return GetTagDataCount(Tsegment); }

		const Tangent	*GetTangentR(void)  {	return (Tangent*)GetTagDataR(Thermite2d); }
		Tangent				*GetTangentW(void) {	return (Tangent*)GetTagDataW(Thermite2d); }
		LONG					GetTangentCount(void) {	return GetTagDataCount(Thermite2d); }

		LineObject   *GetLineObject(BaseDocument *doc, Real lod, BaseThread *thread=NULL);

		static SplineObject *Alloc(LONG pcnt, LONG type);
		static void Free(SplineObject *&bl);
};

BaseObject *GeneratePrimitive(BaseDocument *doc, LONG type, const BaseContainer &bc, Real lod, Bool isoparm, BaseThread *bt=NULL);
BaseObject *GenerateSplinePrimitive(BaseDocument *doc, LONG type, const BaseContainer &bc, Real lod, BaseThread *bt=NULL);

#define ToPoint(op)		((PointObject*)(op))
#define ToPoly(op)		((PolygonObject*)(op))
#define ToSpline(op)  ((SplineObject*)(op))

inline Vector CalcFaceNormal(const Vector *padr, const CPolygon &v)
{
	if (v.c==v.d)
		return !((padr[v.b]-padr[v.a])%(padr[v.c]-padr[v.a]));
	else
		return !((padr[v.b]-padr[v.d])%(padr[v.c]-padr[v.a]));
}

class Safety
{
	private:
		ObjectSafety *os;
	public:
		Safety(void);
		~Safety(void);
		Bool Init(BaseObject *op);
		void Detach(void);
};

LONG CalcLOD(LONG val, Real lod, LONG min, LONG max);
Bool DisjointMesh(PointObject *op);

SplineObject *FitCurve(Vector *padr, LONG pcnt, Real error, BaseThread *bt);
Bool Triangulate(const Vector *padr, LONG pcnt, CPolygon **vadr, LONG *vcnt);
Bool TriangulateStandard(const Vector *padr, LONG pcnt, LONG *list, LONG lcnt, CPolygon *&vadr, LONG &vcnt, BaseThread *thread);
Bool TriangulateRegular(const Vector *pinp, LONG pinp_cnt, LONG *list, LONG lcnt, Vector *&padr, LONG &pcnt, CPolygon *&vadr, LONG &vcnt, Real regular_width, BaseThread *thread);
BaseObject *GenerateText(BaseContainer *cp, BaseThread *bt, Bool separate);
Bool HNSubdivideObject(BaseDocument *doc, BaseObject *pop, LONG sub, LONG subiso, LONG generatebits, BaseThread *thread);
Bool CheckDisplayFilter(BaseObject *op, LONG filter);
LONG IntersectionTest(PolygonObject *op, BaseDraw *bd, Real x, Real y, const Matrix &mg, Real *z, LONG mode, UCHAR *pPointSelect, LONG lSelectCount);
Bool AddTextureString(GetAllStringData *gd, const Filename &fn, BaseList2D *bl);

struct PolyInfo
{
	LONG face[4];
	LONG edge[4];
	CHAR mark[4];
};

struct NgonNeighbor
{
  struct NgonNeighborSegment
  {
    LONG ptcnt;   // number of points in this segment
    LONG *pts;		// points
    LONG *edges;	// edges
    LONG *nbr;		// neighbor polygons
  };
  LONG segcnt; // number of segments (always >= 1)
  NgonNeighborSegment* segments; // always != NULL
};

class Neighbor
{
	private:
		EnumerateEdges *ee;
	public:
		Neighbor(void);
		virtual ~Neighbor(void);

		virtual Bool Init(LONG pcnt, const CPolygon *vadr, LONG vcnt, BaseSelect *bs);
		void GetEdgePolys(LONG a, LONG b,LONG *first,LONG *second) { C4DOS.Nb->GetEdgePolys(ee,a,b,first,second); }
		void GetPointPolys(LONG pnt, LONG **dadr, LONG *dcnt)  { C4DOS.Nb->GetPointPolys(ee,pnt,dadr,dcnt); }
		LONG GetEdgeCount(void);
		PolyInfo *GetPolyInfo(LONG poly) { return C4DOS.Nb->GetPolyInfo(ee,poly); }
		LONG GetNeighbor(LONG a, LONG b, LONG poly);
		Bool GetNGons(PolygonObject* op, LONG &ngoncnt, NgonNeighbor *&ngons) { return C4DOS.Nb->GetNGons(ee, op, ngoncnt, ngons); }
		void Flush() { C4DOS.Nb->Free(ee); ee=NULL; }
		void ResetAddress(const CPolygon *a_polyadr) { C4DOS.Nb->ResetAddress(ee,a_polyadr); }
};

BaseObject *GetVirtualLineObject(BaseObject *op, HierarchyHelp *hh, const Matrix &mloc, Bool keep_spline, Bool recurse, Matrix *mres, Bool *dirty);

UVWTag *GenerateUVW(BaseObject *op, const Matrix &opmg, TextureTag *tp, const Matrix &texopmg, BaseView *view);

// special handling: light object
void CutReal(BaseContainer &data, LONG id, Real min, Real max);
void CutVector(BaseContainer &data, LONG id, Real min, Real max);
Bool CalculateVisiblePoints(BaseDraw *bd, PolygonObject *op, Vector *padr, UCHAR *pset, Bool select_visibonly);

// for triangulating
#define POLY_TRIANG_CONSTRAINED_OUTLINE     1 // all points form the outline, the segments are ignored
#define POLY_TRIANG_QUADS                   8 // untriangulate where possible
#define POLY_TRIANG_NGON                    32 // must be set if an ngon is to be triangulated.
#define POLY_TRIANG_FORCE_VORONOI           64 // force Voronoi triangulation
#define POLY_TRIANG_FORCE_NGON              (128 | POLY_TRIANG_NGON) // force special ngon triangulation
#define POLY_TRIANG_FILL_HOLES							256 // fill segment holes (only used for ngon triangulation)

#define TRIANGSTATE_SKIP_EDGE           1
#define TRIANGSTATE_MISSING_CONSTRAINT  2

class PolyTriangulate
{
	private:
		PolyTriangulate();
		~PolyTriangulate();
	public:
		static PolyTriangulate* Alloc();
		static void Free(PolyTriangulate *&pTriang);

		// Don't store pPolys for later use. It will get invalid after the next Triangulate call
		// or when this class instance is destroyed.
		// Use TriangulateRelease if you need to store the result for a later use.

		// lConstraints and plConstrainedEdges are reserved for later use
		Bool Triangulate(const Vector* pvPoints, LONG lPointCount, const LONG* plSegments, LONG lSegCnt,
			CPolygon*& pPolys, LONG& lPolyCount, LONG lFlags = 0, const LONG* plMap = NULL, BaseThread* pThread = NULL,
			LONG lConstraints = 0, const LONG* plConstrainedEdges = NULL);
		Bool Triangulate(const LVector* pvPoints, LONG lPointCount, const LONG* plSegments, LONG lSegCnt,
			CPolygon*& pPolys, LONG& lPolyCount, LONG lFlags = 0, const LONG* plMap = NULL, BaseThread* pThread = NULL,
			LONG lConstraints = 0, const LONG* plConstrainedEdges = NULL);

		// You must call GeFree(pPolys) to free memory
		Bool TriangulateRelease(const Vector* pvPoints, LONG lPointCount, const LONG* plSegments, LONG lSegCnt,
			CPolygon*& pPolys, LONG& lPolyCount, LONG lFlags = 0, const LONG* plMap = NULL, BaseThread* pThread = NULL,
			LONG lConstraints = 0, const LONG* plConstrainedEdges = NULL);
		Bool TriangulateRelease(const LVector* pvPoints, LONG lPointCount, const LONG* plSegments, LONG lSegCnt,
			CPolygon*& pPolys, LONG& lPolyCount, LONG lFlags = 0, const LONG* plMap = NULL, BaseThread* pThread = NULL,
			LONG lConstraints = 0, const LONG* plConstrainedEdges = NULL);

		Bool HasIdentical();
		void SetPolygonMatrix(LMatrix* m);
		ULONG GetState();
};

#endif
