/////////////////////////////////////////////////////////////
// CINEMA 4D SDK                                           //
/////////////////////////////////////////////////////////////
// (c) 1989-2005 MAXON Computer GmbH, all rights reserved  //
/////////////////////////////////////////////////////////////

#ifndef __LIB_HAIR_H__
#define __LIB_HAIR_H__

#include "c4d_library.h"
#include "c4d_baseobject.h"
#include "c4d_basetag.h"
#include "c4d_videopost.h"

//////////////////////////////////////////////////////////////////////////

class ChainBase;
class HairManager;

//////////////////////////////////////////////////////////////////////////

#define Ohair						1017305
#define Thairselection	1017746
#define Thairvertex			1018329
#define VPhair					1017325
#define Mhair						1017730
#define Thairmaterial		1017729
#define Thaircollider		1017326
#define Thairtangent		1017671
#define Thairlight			1017874

//////////////////////////////////////////////////////////////////////////

#define HAIR_ROOT_TYPE_INVALID	 -1
#define HAIR_ROOT_TYPE_NONE				0
#define HAIR_ROOT_TYPE_VERTEX			1
#define HAIR_ROOT_TYPE_POLY				2
#define HAIR_ROOT_TYPE_SEGMENT		3
#define HAIR_ROOT_TYPE_SPLINE2D		4
#define HAIR_ROOT_TYPE_SPLINE3D		5
#define HAIR_ROOT_TYPE_GUIDES			6
#define HAIR_ROOT_TYPE_NULL				7
#define HAIR_ROOT_TYPE_CLONE			8

#define HAIR_GENERATE_FLAGS_NONE			0
#define HAIR_GENERATE_FLAGS_NO_MATERIAL		(1<<0)
#define HAIR_GENERATE_FLAGS_NO_DYNAMICS		(1<<1)
#define HAIR_GENERATE_FLAGS_NO_TRANSFORM	(1<<2)
#define HAIR_GENERATE_FLAGS_NO_DEFORMERS	(1<<3)

#define HAIR_MODE_LOCKED	-2
#define HAIR_MODE_HIDDEN	-1

#define HAIR_MODE_TIPS		0
#define HAIR_MODE_POINTS	1
#define HAIR_MODE_GUIDES	2
#define HAIR_MODE_ROOTS		3
#define HAIR_MODE_VERTEX	4

#define HAIR_MPARAMETER_THICKNESS			1
#define HAIR_MPARAMETER_COLOR					2
#define HAIR_MPARAMETER_BACKCOLOR			3
#define HAIR_MPARAMETER_TRANSPARENCY	4

#define HAIR_LOCK_FLAGS_ANIMATING			(1<<2)
#define HAIR_LOCK_FLAGS_FULL_UPDATE		(1<<4)
#define HAIR_LOCK_FLAGS_NO_TRANSFORM	(1<<5)

//////////////////////////////////////////////////////////////////////////

class HairGuides;
class HairObject;
class iHairCollider;
class HairColliderData;

struct HairRootData
{
	HairRootData() { m_Type=HAIR_ROOT_TYPE_INVALID; }

	LONG	m_ID,m_Type;
	Real	m_S,m_T;
	Vector	m_P,m_N;
};

struct HairPolygonHit
{
	BaseObject *m_pObject;	// object that was hit

	LONG m_gID;				// guide ID
	LONG m_sID;				// segment ID
	Real m_sT;				// position along segment;

	LONG m_ID;				// polygon index
	Vector m_P;				// hit point

	Vector m_Vin,m_Vout;	// velocity in/out of hit
	Real	m_T;			// frame delta of hit
};

struct RootObjectData
{
	BaseObject *pObject;
	BaseTag *pTag;

	Neighbor *pNeighbor;
	
	const Vector *pPoint;
	const Vector *pOriginPoints;

	const CPolygon *pPolygon;
	const Segment *pSegment;
};

class HairGuideWeight
{
public:
	HairGuideWeight() { m_Index=NOTOK; m_Weight=0.0; }

	LONG m_Index;
	Real m_Weight;
};

class HairInterpolationMap
{
	friend class ChainBase;
	friend class HairManager;

public:
	HairInterpolationMap()
	{
		m_pMap=NULL;
		m_GuideCount=0;
		m_Count=0;
		m_WeightCount=0;
	}
	~HairInterpolationMap()
	{
		Free();
	}

	void Free()
	{
		GeFree(m_pMap);
		m_GuideCount=0;
		m_Count=0;
		m_WeightCount=0;
	}
	
	HairGuideWeight *GetMap() { return m_pMap; }
	HairGuideWeight *GetGuides(LONG index)
	{
		if (!m_pMap) return NULL;
		return &m_pMap[index*m_WeightCount];
	}
	LONG GetHairCount() { return m_Count; }
	LONG GetGuideCount() { return m_GuideCount; }
	LONG GetWeightCount() { return m_WeightCount; }

	LReal GetTotalWeight(LONG index);
	LReal GetTotalInvWeight(LONG index);

	LReal GetInterpolateReal(LONG index, LReal *v);
	Real GetInterpolateReal(LONG index, Real *v);
	Vector GetInterpolatedVector(LONG index, Vector *v);
	LVector GetInterpolatedVector(LONG index, LVector *v);

private:

	HairGuideWeight *m_pMap;

	LONG m_WeightCount;
	LONG m_GuideCount;
	LONG m_Count;
};

//////////////////////////////////////////////////////////////////////////

class HairMaterialData
{
private:
	HairMaterialData();
	~HairMaterialData();
public:
	LONG GetCount();
	LONG GetGuideMaterial(LONG i, LONG prev=NOTOK);

	BaseTag *GetMaterialTag(LONG i);

	Bool ApplyMaterial(LONG i, HairGuides *guides, VolumeData *vd=NULL, LONG vindex=0);
	Bool ApplyMaterials(HairGuides *guides, VolumeData *vd=NULL, LONG vindex=0);

	Real GetThickness(LONG guide, Real t, LONG i=NOTOK);		
	Vector GetColor(LONG guide, Real t, const Vector &r, const Vector &wp, const Vector &p, const Vector &n, VolumeData *vd, const RayHitID &plyid, LONG i=NOTOK);
	Vector GetBackColor(LONG guide, Real t, const Vector &col, const Vector &r, const Vector &wp, const Vector &p, const Vector &n, VolumeData *vd, const RayHitID &plyid, LONG i=NOTOK);
	Real GetTransparency(LONG guide, Real t, const Vector &p, const Vector &n, LONG i=NOTOK);

	Bool GetTagSelection(LONG i, BaseSelect *bs);
};

class HairGuideDynamics
{
private:
	HairGuideDynamics();
	~HairGuideDynamics();
public:
	HairGuides *GetGuides();

	Vector GetPosition(LONG i);
	void SetPosition(LONG i, const Vector &p);

	Vector GetLastPosition(LONG i);
	void SetLastPosition(LONG i, const Vector &p);

	Vector GetVelocity(LONG i);
	void SetVelocity(LONG i, const Vector &v);

	Real GetMass(LONG i);
	void SetMass(LONG i, Real m);

	Vector GetForce(LONG i);
	void AddForce(LONG i, const Vector &f);

	LONG GetPolygonIntersections(HairPolygonHit **hits);	// returned 'hits' need to be free'd with GeFree, returns the number of hits or NOTOK for an error
};

#define HAIR_GUIDE_FLAGS_LOCAL		(1<<0)	// chains are in local object space
#define HAIR_GUIDE_FLAGS_AS_GUIDES	(1<<1)	// special case guides
#define HAIR_GUIDE_FLAGS_INITIAL	(1<<3)	// chains are in their original initial space (like a sticktexturetag)
#define HAIR_GUIDE_FLAGS_HAIRS		(1<<4)	// created for hairs
#define HAIR_GUIDE_FLAGS_INITIALAXIS	(1<<6)	// initial space and root axis aligned

class HairGuides
{
private:
	HairGuides();
	~HairGuides();
public:
	static HairGuides *Alloc(LONG count, LONG segments);
	static void Free(HairGuides *&guides);

	LONG GetCount();	// number of guides
	LONG GetSegmentCount();	// number of segments per guide (number of points is +1)
	LONG GetPointCount() { return GetCount()*GetGuidePointCount(); }
	LONG GetGuidePointCount() { LONG s=GetSegmentCount(); return (s==0)?0:(s+1); }	// number of points per segment

	Vector *GetPoints();	// get a pointer to the point vectors for the guides
	HairGuideDynamics *GetDynamics();	// get the dynamics (can be NULL)

	Matrix GetMg();
	void SetMg(const Matrix &mg);

	HairRootData GetRoot(LONG index);
	void SetRoot(LONG index, const HairRootData &root, Bool update);

	HairObject *GetObject();

	Bool GetSelected(LONG mode, BaseSelect *select);
	Bool SetSelected(LONG mode, BaseSelect *select);
	Bool ConvertSelection(LONG from_mode, LONG to_mode, BaseSelect *from_select, BaseSelect *to_select);

	Bool CopyFrom(const HairGuides *src);

	Vector GetTangent(LONG guide, LONG segment, Real t);

	SplineObject *CreateSpline();

 	void ToLocal();
	void ToWorld();
	void ToInitial(Bool align);
	
	void UndisplaceRoots();
	void DisplaceRoots();

	Bool GetRootAxis(LONG index, Matrix &m, Bool bAlign=FALSE, Bool bLocal=TRUE, Bool bInitial=FALSE, Bool bZAxis=FALSE);
	Vector GetRootUV(LONG index);

	Bool GetRootData(LONG index, Vector *p=NULL, Vector *n=NULL, LONG *ply_id=NULL, Bool bLocal=TRUE, Bool bDisplaced=TRUE, Bool bInitial=FALSE);

	Matrix *GetTransformMatrix();
	void SetTransformMatrix(Matrix *tm);
  
	Bool Delete(BaseSelect *bs);
	Bool AddRoot(LONG ply_id, Real s, Real t, Real len);

	Bool GetRootObject(RootObjectData *pData);

	Bool LinkToObject(HairObject *pHair);

	LONG GetFlags();
	LONG SetFlags(LONG flags);
};

class HairObject : public BaseObject
{
private:
	HairObject();
	~HairObject();
public:
	static HairObject *Alloc() { return (HairObject*)BaseObject::Alloc(Ohair); }
	static void Free(HairObject *&pObject) { BaseObject *op=pObject; BaseObject::Free(op); pObject=NULL; }

	HairGuides *GetGuides();
	HairGuides *GetDynamicGuides();
	HairGuides *GenerateHair(LONG flags=HAIR_GENERATE_FLAGS_NONE, LONG count=NOTOK, LONG segments=NOTOK, HairMaterialData **pMaterial=NULL, HairInterpolationMap *pMap=NULL);

	Bool GetRootObject(BaseObject **pObject, BaseTag **pTag, RootObjectData *pData);

	Bool Lock(BaseDocument *pDoc, BaseThread *pThread, Bool bValidate, LONG flags);
	Bool IsLocked();
	void Unlock();

	Bool SetGuides(HairGuides *guides, Bool clone);
	void RemoveGuides();

	Bool Update(BaseDocument *doc);	// do NOT use within a Lock()

	HairMaterialData *InitMaterials(BaseDocument *pDoc, HairGuides *guides=NULL, InitRenderStruct *irs=NULL, VolumeData *vd=NULL);
	void FreeMaterials(HairMaterialData *&pMaterials);
};

class HairVertexMapTag : public PluginTag
{
private:
	HairVertexMapTag();
	~HairVertexMapTag();
public:
	static HairVertexMapTag *Alloc() { return (HairVertexMapTag*)PluginTag::Alloc(Thairvertex); }
	static void Free(HairVertexMapTag *&pTag) { PluginTag *tag=pTag; PluginTag::Free(tag); pTag=NULL; }

	LONG GetCount();
	LONG GetSegments();
	LONG GetPointCount();
	UWORD *GetMap();				// to get the real value divide by 65535.0
};

class HairSelectionTag : public PluginTag
{
private:
	HairSelectionTag();
	~HairSelectionTag();
public:
	static HairSelectionTag *Alloc() { return (HairSelectionTag*)PluginTag::Alloc(Thairselection); }
	static void Free(HairSelectionTag *&pTag) { PluginTag *tag=pTag; PluginTag::Free(tag); pTag=NULL; }

	Bool GetSelected(BaseSelect *bs);
	Bool SetSelected(BaseSelect *bs);

	LONG GetType();
	void SetType(LONG mode);

	LONG GetCount();
	LONG GetSegments();
};

class HairTangentTag : public PluginTag
{
private:
	HairTangentTag();
	~HairTangentTag();
public:
	static HairTangentTag *Alloc() { return (HairTangentTag*)PluginTag::Alloc(Thairtangent); }
	static void Free(HairTangentTag *&pTag) { PluginTag *tag=pTag; PluginTag::Free(tag); pTag=NULL; }

	LONG GetPolygonsSegments();
	LONG GetPointCount();
	LONG GetCount();
	LONG GetSegments();
	Vector *GetTangent();
};

class HairRenderObject
{
public:
	HairObject *m_pGuideObject;
	HairObject *m_pHairObject;
	HairGuides *m_pHair;
	BaseList2D *m_pBase;
	RayObject *m_pRayObject;
	HairMaterialData *m_pMaterial;
	Real		*m_pThickness;	
};

struct HairFragment
{
	Vector col;
	Real trans;
	Real z;
};

struct HairFragmentLink
{
	Vector col;
	Real trans;
	Real z;
	void *link;
};

struct HairHitData
{
	LONG oindex;
	LONG lid;
	LONG seg;
	LONG p;
	HairGuides *hair;
	BaseList2D *object;
};

#define HAIR_VP_FLAG_NOHOOKS	(1<<0)

class HairVideoPost : public PluginVideoPost
{
private:
	HairVideoPost();
	~HairVideoPost();
public:
	static HairVideoPost *Alloc() { return (HairVideoPost*)PluginVideoPost::Alloc(VPhair); }
	static void Free(HairVideoPost *&pPost) { PluginVideoPost *vp=pPost; PluginVideoPost::Free(vp); pPost=NULL; }

	LONG GetObjectCount();
	void GetObject(LONG i, HairRenderObject *hro);
	LONG FindObject(BaseList2D *bl);

	Bool TracerEnabled();
	LONG TraceRay(const Vector &p, const Vector &v, Vector &hit, Real &d, LONG cpu, Real tf, LONG flags);

	void Sample(LONG oindex, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &v, Vector &col, const Vector &n, const Vector &lp, const Vector &t, const Vector &r, Vector huv, LONG flags);
	Real SampleTransparency(LONG oindex, VolumeData *vd, LONG lid, LONG seg, LONG p, Real lined, const Vector &n, const Vector &linep, const Vector &lp, Vector huv, LONG cpu, LONG flags, RayLight *light);
	Real SampleShadow(VolumeData *vd, RayLight *light, const Vector &p, Real delta, LONG cpu, LONG flags);

	Bool GetTopFragment(LONG x, LONG y, HairFragment *frag, LONG cpu);
	Bool GetFragmentLink(LONG x, LONG y, HairFragmentLink *frag, LONG cpu, Bool first);
	HairFragment *GetFragments(LONG x, LONG y, LONG &cnt, LONG cpu);
	Bool SetFragments(LONG x, LONG y, HairFragment *frag, LONG cnt, LONG cpu);

	Bool InsertFragment(LONG x, LONG y, HairFragment *frag, LONG cnt, LONG cpu);
	Bool InsertFragmentLine(LONG x, LONG y, LONG xcnt, HairFragment **frag, LONG *cnt, LONG cpu);

	void GetHitInfo(LONG id, HairHitData *hdata);
};

//////////////////////////////////////////////////////////////////////////

#define MSG_HAIR_GET_OBJECT_TYPE	1018952		// HairPluginMessageData
#define MSG_HAIR_GET_OBJECTS		1018959		// AtomArray

#define HAIR_CALC_FLAG_APPLYMATERIALS	(1<<0)
#define HAIR_CALC_FLAG_APPLYDEFORMERS	(1<<1)
#define HAIR_CALC_FLAG_DISPLACE			(1<<2)
#define HAIR_CALC_FLAG_SKIP				(1<<3)
#define HAIR_CALC_FLAG_NOMEMORY			(1<<4)
#define HAIR_CALC_FLAG_BREAK			(1<<5)

#define HAIR_PLUGIN_TYPE_FORCE		1
#define HAIR_PLUGIN_TYPE_CONSTRAINT	2
#define HAIR_PLUGIN_TYPE_COLLIDER	3
#define HAIR_PLUGIN_TYPE_DEFORMER	4
#define HAIR_PLUGIN_TYPE_GENERATOR	5
#define HAIR_PLUGIN_TYPE_SHADER		6
#define HAIR_PLUGIN_TYPE_STYLING	7

#define HAIR_GENERATE_PASS_INIT			0
#define HAIR_GENERATE_PASS_BUILD		1
#define HAIR_GENERATE_PASS_FREE			2
#define HAIR_GENERATE_PASS_THICKNESS	3

#define HAIR_GENERATE_HOOK_BUILD		1
#define HAIR_GENERATE_HOOK_DYNAMICS		2

#define HAIR_HOOK_TYPE_SAMPLE_COLOR		1
#define HAIR_HOOK_TYPE_SAMPLE_TRANS		2
#define HAIR_HOOK_TYPE_SAMPLE_SHADOWS	3
#define HAIR_HOOK_TYPE_GENERATE_HAIR	4
#define HAIR_HOOK_TYPE_ILLUMINATE		5

#define HAIR_RENDER_INIT_PASS_PRE		1
#define HAIR_RENDER_INIT_PASS_BUILD		2
#define HAIR_RENDER_INIT_PASS_POST		3

typedef Bool (*HrDeform)(BaseDocument *doc, BaseList2D *op, HairObject *hair, HairGuides *guides, Vector *padr, LONG cnt, LONG scnt);
typedef Bool (*HrStyle)(BaseDocument *doc, BaseList2D *op, HairObject *hair, HairGuides *guides, Vector *padr, LONG cnt, LONG scnt);
typedef Bool (*HrForce)(BaseDocument *doc, BaseList2D *op, HairObject *hair, HairGuides *guides, HairGuideDynamics *dyn, Vector *force, Real *invmass, Vector *padr, LONG pcnt, LONG cnt, LONG scnt, Real t1, Real t2);
typedef Bool (*HrConstraint)(BaseDocument *doc, BaseList2D *op, HairObject *hair, HairGuides *guides, HairGuideDynamics *dyn, Vector *oldpnt, Vector *newpnt, Real *invmass, LONG pcnt, LONG cnt, LONG scnt);
typedef Bool (*HrCollision)(BaseDocument *doc, BaseList2D *op, HairObject *hair, HairGuides *guides, HairGuideDynamics *dyn, const Vector &bmin, const Vector &bmax, Real t1, Real t2, Real pr, Vector *oldpnt, Vector *newpnt, Vector *vel, Real *invmass, LONG pcnt, LONG cnt, LONG scnt);

typedef HairGuides *(*HrGenerate)(BaseDocument *doc, BaseList2D *op, HairObject *hair, BaseThread *thd, VolumeData *vd, LONG pass, void *data);
typedef LONG (*HrCalcHair)(LONG index, LONG oindex, NodeData *node, HairGuides *guides, Vector *guide_pnts, Vector *rend_pnts, Real *thickness, VolumeData *vd, Vector *n);
typedef LONG (*HrGenerateAdd)(BaseDocument *doc, BaseList2D *op);
typedef Vector (*HrGenerateColor)(HairVideoPost *vp, LONG oindex, HairMaterialData *mat, RayObject *ro, HairObject *op, HairGuides *guides, BaseList2D *bl, Real *thk, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &v, const Vector &n, const Vector &lp, const Vector &t, const Vector &r, const Vector &huv, const RayHitID &ply_hit);
typedef Real (*HrGenerateTransparency)(HairVideoPost *vp, LONG oindex, HairMaterialData *mat, RayObject *ro, HairObject *op, HairGuides *guides, BaseList2D *bl, Real *thk, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &n, const Vector &lp, const Vector &huv, const RayHitID &ply_hit, RayLight *light);

typedef Vector (*HrSampleExt)(PluginShader *shader, NodeData *node, ChannelData* cd, HairGuides *guides, LONG i, Real t);

typedef Vector (*HrSampleHairColorHook)(HairVideoPost *vp, LONG oindex, HairMaterialData *mat, RayObject *ro, HairObject *op, HairGuides *guides, BaseList2D *bl, Real *thk, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &v, const Vector &n, const Vector &lp, const Vector &t, const Vector &r, const Vector &huv, const RayHitID &ply_hit);
typedef Real (*HrSampleHairTransparencyHook)(HairVideoPost *vp, LONG oindex, HairMaterialData *mat, RayObject *ro, HairObject *op, HairGuides *guides, BaseList2D *bl, Real *thk, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &n, const Vector &lp, const Vector &huv, const RayHitID &ply_hit, RayLight *light);
typedef Real (*HrSampleShadowBufferHook)(HairVideoPost *vp, VolumeData *vd, RayLight *light, const Vector &p, Real delta, LONG cpu);

typedef Bool (*HrGenerateHairHook)(HairObject *op, HairGuides *guides, HairGuides *hair, LONG pass);
typedef Bool (*HrIlluminateHook)(HairVideoPost *vp, VolumeData *vd, RayLight *light, Vector &colshad, Vector &col, Vector &lv, const Vector &p, const Vector &v);

typedef Bool (*HrInitRender)(HairVideoPost *vp, VolumeData *vd, BaseDocument *doc, BaseList2D *bl, HairObject *op, HairGuides *guides, LONG oindex, LONG pass);
typedef void (*HrFreeRender)(HairVideoPost *vp, BaseList2D *bl);

typedef Real (*HrModifyHairShadowTransparency)(HairVideoPost *vp, LONG oindex, HairMaterialData *mat, RayObject *ro, HairObject *op, HairGuides *guides, BaseList2D *bl, Real *thk, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &n, const Vector &lp, const Vector &huv, const RayHitID &ply_hit, RayLight *light, Real trans);

#define HAIR_PLUGIN_DATA_VERSION	1000

class HairPluginObjectData
{
public:
	HairPluginObjectData()
	{
		ClearMem(this,sizeof(HairPluginObjectData));
		version=HAIR_PLUGIN_DATA_VERSION;
		size=sizeof(HairPluginObjectData);
	}

	LONG version,size;
	LONG flags;

	//////////////////////////////////////////////////////////////////////////

	HrForce			calc_force;
	HrConstraint	calc_constraint;
	HrCollision		calc_collision;
	HrDeform		calc_deform;
	HrGenerate		calc_generate;
	HrGenerateAdd	add_generate;
	HrCalcHair		calc_hair;
	HrSampleExt		calc_sample;
	HrGenerateColor	calc_col;
	HrGenerateTransparency	calc_trans;
	HrStyle			calc_style;
	HrInitRender	init_render;
	HrFreeRender	free_render;
	HrModifyHairShadowTransparency calc_shad;
};

#define HAIR_MSG_DATA_VERSION	1000

class HairPluginMessageData
{
public:
	HairPluginMessageData()
	{
		size=sizeof(HairPluginMessageData);
		version=HAIR_MSG_DATA_VERSION;
		data=NULL;
		flags=0;
	}
	
	LONG	version,size;		// READ ONLY
	
	LONG	flags;
	void	*data;
};

class HairLibrary
{
public:
	LONG GetMode(BaseDocument *doc);
	void SetMode(BaseDocument *doc, LONG mode);

	Bool GetHairGlobal(BaseDocument* doc);
	void SetHairGlobal(BaseDocument* doc, Bool enable);

	Vector BlendColors(LONG mode, const Vector &colA, const Vector &colB);

	Vector MixST(Real s, Real t, const Vector &pa, const Vector &pb, const Vector &pc, const Vector &pd, Bool bQuad);
	Real MixST(Real s, Real t, Real va, Real vb, Real vc, Real vd, Bool bQuad);
	void GetPolyPointST(const Vector &p, Real &s, Real &t, const Vector &pa, const Vector &pb, const Vector &pc, const Vector &pd, Bool bQuad);

	void *GetHook(BaseDocument *doc, LONG type);
	void *SetHook(BaseDocument *doc, LONG type, void *fn);

	BaseContainer *GetPrefsInstance();

	LONG GetHairVersion();
};

//////////////////////////////////////////////////////////////////////////

class HairCollider
{
private:
	HairColliderData *m_pData;

	HairCollider();
	~HairCollider();
public:
	static HairCollider *Alloc();
	static void Free(HairCollider *&p);

	Bool Init(HairGuides *hair);
	void Release();

	LONG GetClosestPoint(const Vector &p);
	Bool GetClosestSegment(const Vector &p, LONG *guide, LONG* segment, Real* segt);
	Bool GetParticleIntersection(const Vector &p, const Vector &v, Real r, LONG *guide, LONG *segment, Real *segt, Real *rayt);
};

// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF

//////////////////////////////////////////////////////////////////////////
#define LIBRARY_HAIR			1018173
#define LIBRARY_HAIR_COLLIDER	1018174
//////////////////////////////////////////////////////////////////////////

struct HairObjectLib : public C4DLibrary
{
	HairGuides *(*GetGuides)(BaseObject *hop);
	HairGuides *(*GetDynamicGuides)(BaseObject *hop);
	Bool (*GetRootObject)(BaseObject *hop, BaseObject **pObject, BaseTag **pTag, RootObjectData *pData);
	Bool (*Lock)(BaseObject *hop, BaseDocument *pDoc, BaseThread *pThread, Bool bValidate, LONG flags);
	void (*Unlock)(BaseObject *hop);
	HairGuides *(*GenerateHair)(BaseObject *hop, LONG flags, LONG count, LONG segments, HairMaterialData **pMaterial, HairInterpolationMap *pMap);
	Bool (*SetGuides)(BaseObject *hop, HairGuides *guides, Bool clone);
	void (*RemoveGuides)(BaseObject *hop);
	HairMaterialData *(*InitMaterials)(BaseObject *hop, BaseDocument *pDoc, HairGuides *guides, InitRenderStruct *irs, VolumeData *vd);
	void (*FreeMaterials)(BaseObject *hop, HairMaterialData *&pMaterials);
	LONG (*GetGuideCount)(HairGuides *guides);
	LONG (*GetGuideSegmentCount)(HairGuides *guides);
	Vector *(*GetGuidePoints)(HairGuides *guides);
	HairGuideDynamics *(*GetGuideDynamics)(HairGuides *guides);
	Matrix (*GetGuideMg)(HairGuides *guides);
	void (*SetGuideMg)(HairGuides *guides, const Matrix &mg);
	HairRootData (*GetGuideRoot)(HairGuides *guides, LONG index);
	void (*SetGuideRoot)(HairGuides *guides, LONG index, const HairRootData &root, Bool update);
	HairObject *(*GetGuideObject)(HairGuides *guides);
	HairGuides *(*AllocGuides)(LONG count, LONG segments);
	void (*FreeGuides)(HairGuides *&guides);
	Bool (*GetGuideSelected)(HairGuides *guides, LONG mode, BaseSelect *select);
	Bool (*CopyGuidesFrom)(HairGuides *guides, const HairGuides *src);
	Bool (*SetGuideSelected)(HairGuides *guides, LONG mode, BaseSelect *select);
	Bool (*ConvertGuideSelection)(HairGuides *guides, LONG from_mode, LONG to_mode, BaseSelect *from_select, BaseSelect *to_select);
	Vector (*GetGuideTangent)(HairGuides *guides, LONG guide, LONG segment, Real t);
	SplineObject *(*CreateGuideSpline)(HairGuides *guides);
	void (*guideToLocal)(HairGuides *guides);
	void (*guideToWorld)(HairGuides *guides);
	void (*guideToInitial)(HairGuides *guides, Bool align);
	void (*guideUndisplaceRoots)(HairGuides *guides);
	void (*guideDisplaceRoots)(HairGuides *guides);
	Bool (*guideGetRootAxis)(HairGuides *guides, LONG index, Matrix &m, Bool bAlign, Bool bLocal, Bool bInitial, Bool bZAxis);
	Vector (*guideGetRootUV)(HairGuides *guides, LONG index);
	Bool (*guideGetRootData)(HairGuides *guides, LONG index, Vector *p, Vector *n, LONG *ply_id, Bool bLocal, Bool bDisplaced, Bool bInitial);
	Matrix *(*guideGetTransformMatrix)(HairGuides *guides);
	void (*guideSetTransformMatrix)(HairGuides *guides, Matrix *tm);
	Bool (*guideDelete)(HairGuides *guides, BaseSelect *bs);
	Bool (*guideAddRoot)(HairGuides *guides, LONG ply_id, Real s, Real t, Real len);
	Bool (*guideGetRootObject)(HairGuides *guides, RootObjectData *pData);
	HairGuides *(*GetDynamicsGuides)(HairGuideDynamics *dynamics);
	Vector (*GetDynamicsPosition)(HairGuideDynamics *dynamics, LONG i);
	void (*SetDynamicsPosition)(HairGuideDynamics *dynamics, LONG i, const Vector &p);
	Vector (*GetDynamicsLastPosition)(HairGuideDynamics *dynamics, LONG i);
	void (*SetDynamicsLastPosition)(HairGuideDynamics *dynamics, LONG i, const Vector &p);
	Vector (*GetDynamicsVelocity)(HairGuideDynamics *dynamics, LONG i);
	void (*SetDynamicsVelocity)(HairGuideDynamics *dynamics, LONG i, const Vector &v);
	Real (*GetDynamicsMass)(HairGuideDynamics *dynamics, LONG i);
	void (*SetDynamicsMass)(HairGuideDynamics *dynamics, LONG i, Real m);
	Vector (*GetDynamicsForce)(HairGuideDynamics *dynamics, LONG i);
	void (*AddDynamicsForce)(HairGuideDynamics *dynamics, LONG i, const Vector &f);
	LONG (*GetDynamicsPolygonIntersections)(HairGuideDynamics *dynamics, HairPolygonHit **hits);
	LONG (*hairGetMode)(BaseDocument *doc);
	void (*hairSetMode)(BaseDocument *doc, LONG mode);
	Bool (*hairGetHairGlobal)(BaseDocument* doc);
	void (*hairSetHairGlobal)(BaseDocument* doc, Bool enable);
	Vector (*hairBlendColors)(LONG mode, const Vector &colA, const Vector &colB);
	Vector (*hairMixST)(Real s, Real t, const Vector &pa, const Vector &pb, const Vector &pc, const Vector &pd, Bool bQuad);
	Real (*hairMixSTReal)(Real s, Real t, Real va, Real vb, Real vc, Real vd, Bool bQuad);
	void (*hairGetPolyPointST)(const Vector &p, Real &s, Real &t, const Vector &pa, const Vector &pb, const Vector &pc, const Vector &pd, Bool bQuad);
	Bool (*HairSelectionGetSelected)(HairSelectionTag *stag, BaseSelect *bs);
	Bool (*HairSelectionSetSelected)(HairSelectionTag *stag, BaseSelect *bs);
	LONG (*HairSelectionGetType)(HairSelectionTag *stag);
	void (*HairSelectionSetType)(HairSelectionTag *stag, LONG mode);
	LONG (*HairSelectionGetCount)(HairSelectionTag *stag);
	LONG (*HairSelectionGetSegments)(HairSelectionTag *stag);
	LONG (*HairVertexGetCount)(HairVertexMapTag *vtag);
	LONG (*HairVertexGetSegments)(HairVertexMapTag *vtag);
	LONG (*HairVertexGetPointCount)(HairVertexMapTag *vtag);
	UWORD *(*HairVertexGetMap)(HairVertexMapTag *vtag);
	LONG (*matGetCount)(HairMaterialData *mat);
	LONG (*matGetGuideMaterial)(HairMaterialData *mat, LONG i, LONG prev);
	BaseTag *(*matGetMaterialTag)(HairMaterialData *mat, LONG i);
	Bool (*matApplyMaterial)(HairMaterialData *mat, LONG i, HairGuides *guides, VolumeData *vd, LONG vindex);
	Bool (*matApplyMaterials)(HairMaterialData *mat, HairGuides *guides, VolumeData *vd, LONG vindex);
	Real (*matGetThickness)(HairMaterialData *mat, LONG guide, Real t, LONG i);
	Vector (*matGetColor)(HairMaterialData *mat, LONG guide, Real t, const Vector &r, const Vector &wp, const Vector &p, const Vector &n, VolumeData *vd, const RayHitID &plyid, LONG i);
	Vector (*matGetBackColor)(HairMaterialData *mat, LONG guide, Real t, const Vector &col, const Vector &r, const Vector &wp, const Vector &p, const Vector &n, VolumeData *vd, const RayHitID &plyid, LONG i);
	Real (*matGetTransparency)(HairMaterialData *mat, LONG guide, Real t, const Vector &p, const Vector &n, LONG i);
	Bool (*matGetTagSelection)(HairMaterialData *mat, LONG i, BaseSelect *bs);
	LONG (*HairTangentGetPolygonsSegments)(HairTangentTag *ttag);
	LONG (*HairTangentGetPointCount)(HairTangentTag *ttag);
	LONG (*HairTangentGetCount)(HairTangentTag *ttag);
	LONG (*HairTangentGetSegments)(HairTangentTag *ttag);
	Vector *(*HairTangentGetTangent)(HairTangentTag *ttag);
	void *(*hairGetHook)(BaseDocument *doc, LONG type);
	void *(*hairSetHook)(BaseDocument *doc, LONG type, void *fn);
	BaseContainer *(*hairGetPrefsInstance)();
	LONG (*vpGetObjectCount)(HairVideoPost *vp);
	void (*vpGetObject)(HairVideoPost *vp, LONG i, HairRenderObject *hro);
	LONG (*vpFindObject)(HairVideoPost *vp, BaseList2D *bl);
	Bool (*vpTracerEnabled)(HairVideoPost *vp);
	LONG (*vpTraceRay)(HairVideoPost *vp, const Vector &p, const Vector &v, Vector &hit, Real &d, LONG cpu, Real tf, LONG flags);
	Bool (*hairIsLocked)(HairObject *op);
	Bool (*guideLinkToObject)(HairGuides *guide, HairObject *pHair);
	void (*vpSample)(HairVideoPost *vp, LONG oindex, VolumeData *vd, LONG cpu, LONG lid, LONG seg, LONG p, Real lined, const Vector &linep, const Vector &v, Vector &col, const Vector &n, const Vector &lp, const Vector &t, const Vector &r, Vector huv, LONG flags);
	Real (*vpSampleTransparency)(HairVideoPost *vp, LONG oindex, VolumeData *vd, LONG lid, LONG seg, LONG p, Real lined, const Vector &n, const Vector &linep, const Vector &lp, Vector huv, LONG cpu, LONG flags, RayLight *light);
	Real (*vpSampleShadow)(HairVideoPost *vp, VolumeData *vd, RayLight *light, const Vector &p, Real delta, LONG cpu, LONG flags);
	Bool (*vpGetTopFragment)(HairVideoPost *vp, LONG x, LONG y, HairFragment *frag, LONG cpu);
	Bool (*vpSetFragments)(HairVideoPost *vp, LONG x, LONG y, HairFragment *frag, LONG cnt, LONG cpu);
	Bool (*vpGetFragmentLink)(HairVideoPost *vp, LONG x, LONG y, HairFragmentLink *frag, LONG cpu, Bool first);
	HairFragment *(*vpGetFragments)(HairVideoPost *vp, LONG x, LONG y, LONG &cnt, LONG cpu);
	Bool (*vpInsertFragment)(HairVideoPost *vp, LONG x, LONG y, HairFragment *frag, LONG cnt, LONG cpu);
	Bool (*vpInsertFragmentLine)(HairVideoPost *vp, LONG x, LONG y, LONG xcnt, HairFragment **frag, LONG *cnt, LONG cpu);
	LONG (*hairGetHairVersion)();
	LONG (*guideGetFlags)(HairGuides *guide);
	LONG (*guideSetFlags)(HairGuides *guide, LONG flags);
	void (*vpGetHitInfo)(HairVideoPost *vp, LONG id, HairHitData *hdata);
};

struct HairColliderLib : public C4DLibrary
{
	iHairCollider *(*Alloc)();
	void (*Free)(iHairCollider *&p);

	Bool (iHairCollider::*Init)(HairGuides *hair);
	void (iHairCollider::*Release)();
	LONG (iHairCollider::*GetClosestPoint)(const Vector &p);
	Bool (iHairCollider::*GetClosestSegment)(const Vector &p, LONG *guide, LONG* segment, Real* segt);
	Bool (iHairCollider::*GetParticleIntersection)(const Vector &p, const Vector &v, Real r, LONG *guide, LONG *segment, Real *segt, Real *rayt);
};

// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF
// INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF -- INTERNAL STUFF

#endif	// __LIB_HAIR_H__
