﻿// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('ntp', function() {
  'use strict';

  var TilePage = ntp.TilePage;

  function hashCode(strKey) {
      var hash = 0;
      if (strKey) {
          for (var i = 0; i < strKey.length; i++) {
              hash = hash * 31 + strKey.charCodeAt(i);
              hash = intValue(hash);
          }
      }
      return hash;
  }
  function intValue(num) {
      var MAX_VALUE = 0x7fffffff;
      var MIN_VALUE = -0x80000000;
      if (num > MAX_VALUE || num < MIN_VALUE) {
          return num &= 0xFFFFFFFF;
      }
      return num;
  }

  /**
   * Creates a new Most Visited object for tiling.
   * @constructor
   * @extends {HTMLAnchorElement}
   */
  function MostVisited() {
    var el = cr.doc.createElement('a');
    el.__proto__ = MostVisited.prototype;
    el.initialize();

    return el;
  }

  MostVisited.prototype = {
    __proto__: HTMLAnchorElement.prototype,

    initialize: function() {
      this.reset();

      this.addEventListener('click', this.handleClick_.bind(this));
      this.addEventListener('keydown', this.handleKeyDown_.bind(this));
      //this.addEventListener('mouseover',this.handleOver_.bind(this));
      //this.addEventListener('mouseout',this.handleOut_.bind(this));
    },

    get index() {
      assert(this.tile);
      return this.tile.index;
    },

    /**
     * Clears the DOM hierarchy for this node, setting it back to the default
     * for a blank thumbnail.
     */
    reset: function() {
      this.className = 'most-visited filler real';
      this.innerHTML =
            '<span class="thumbnail-wrapper fills-parent">' +
              '<span class="thumbnail fills-parent">' +
                 '<div class="thumbnail-title"></div>' +
              '</span>' +
              '<div class="edit-bar">' +
                '<div class="title"></div>'+
                '<div class="pin"></div>' +
                '<div class="edit"></div>' +
                '<div class="remove"></div>' +
              '</div>' +
            '</span>';

      this.querySelector('.remove').title =
          loadTimeData.getString('removethumbnailtooltip');
      this.querySelector('.edit').title = 
          loadTimeData.getString('editThis');

      $("new_title").placeholder = loadTimeData.getString('editInputTitle');
      $("new_url").placeholder = loadTimeData.getString('editInputUrl');
      this.tabIndex = -1;
      this.data_ = null;
      this.href = "#";
    },

    /**
     * Update the appearance of this tile according to |data|.
     * @param {Object} data A dictionary of relevant data for the page.
     */
    updateForData: function(data) {
      if (!data || data.filler) {
        if (this.data_) this.reset();
        return;
      }

      this.data_ = data;
      //this.tabIndex = 0;
      this.classList.remove('filler');

      var thumbnailTitle = this.querySelector('.thumbnail-title');
      thumbnailTitle.textContent = data.title;
      thumbnailTitle.dir = data.direction;
      if (data.defaultlogo) {
          thumbnailTitle.classList.add('defaultlogo-title');
          thumbnailTitle.style.visibility = "visible";
      } else {
          if (data.uselogo) {
              thumbnailTitle.style.visibility = "hidden";
          } else {
              thumbnailTitle.style.visibility = "visible";
          }
      }


      var title = this.querySelector('.title');
      title.style.width = this.clientWidth - 75 + "px";//(16+4)*3 + 5 +5
      title.textContent = data.title ? data.title : data.url;
      title.dir = data.direction;

      var thumbnailUrl = data.thumbnailUrl || 'chrome://thumb/' + data.url;
      if (thumbnailUrl.indexOf("?")>0) {
         thumbnailUrl += "&L0g0Seq="+Date.now();
      }else {
         thumbnailUrl += "?L0g0Seq="+Date.now();
      }
      var thumbnail = this.querySelector('.thumbnail');
      for (var i = 0; i < 5; i++) {
          thumbnail.classList.remove('defaultlogo-' + i);
      }
      if (data.defaultlogo) {
          thumbnail.classList.add('defaultlogo-' + Math.abs(hashCode(data.url) % 5));
          thumbnail.removeAttribute('style');
      } else {
          thumbnail.style.backgroundImage = url(thumbnailUrl);
      }
      this.href = data.url;
      this.updatePinnedState_();
    },

    /**
     * Handles a click on the tile.
     * @param {Event} e The click event.
     */
    handleClick_: function(e) {
      var target = e.target;
      if (target.classList.contains('pin')) {
        this.togglePinned_();
        e.preventDefault();
        e.stopPropagation();
      } else if (target.classList.contains('remove')) {
        if(this.data_.pinned){
          this.togglePinned_();
        }
        this.href = "#";
        this.blacklist_();
        e.preventDefault();
        e.stopPropagation();
      } else if(target.classList.contains('edit')){
            $("u_title").textContent = loadTimeData.getString('editThis');
            $("new_title").value = this.data_.title;
            $("new_url").value = this.data_.url;
            currentEdit = this.data_.url;
            var user_add = $("user-add-container");
            user_add.style.display = "block";
            user_add.style.left = (document.body.offsetWidth - user_add.clientWidth)/2 + "px";
            user_add.style.top = (document.body.offsetHeight - user_add.clientHeight)/2 + "px";
            chrome.send("getMostVisitedForSelect");
            chrome.send("getOpenedURLs");
            currentTileIndex = this.index;
            $("add-over-layer").style.display ="block";
            e.preventDefault();
      }
      else if(this.classList.contains('filler')){
            $("add-over-layer").style.display ="block";
            $("u_title").textContent = loadTimeData.getString('editAddUrl');
            $("new_title").value = "";
            $("new_url").value = "";
            var user_add = $("user-add-container");
            user_add.style.display = "block";
            user_add.style.left = (document.body.offsetWidth - user_add.clientWidth)/2 + "px";
            user_add.style.top = (document.body.offsetHeight - user_add.clientHeight)/2 + "px";
            chrome.send("getMostVisitedForSelect");
            chrome.send("getOpenedURLs");
            currentTileIndex = this.index;
      }
    },

    /**
     * Allow blacklisting most visited site using the keyboard.
     */
    handleKeyDown_: function(e) {
      if (!IS_MAC && e.keyCode == 46 || // Del
      IS_MAC && e.metaKey && e.keyCode == 8) { // Cmd + Backspace
        this.blacklist_();
      }
    },

    handleOver_:function (e) {
        var wrapper = this.querySelector('.thumbnail-wrapper');
        wrapper.classList.add("thumbnail-wrapper-hover");
    },

    handleOut_:function (e) {
        var wrapper = this.querySelector('.thumbnail-wrapper');
        wrapper.classList.remove("thumbnail-wrapper-hover");
    },
    /**
     * Changes the visual state of the page and updates the model.
     */
    togglePinned_: function(pinned) {
      var data = this.data_;
      data.pinned = !data.pinned || pinned;
      if (data.pinned) {
        console.log(" ---------- chrome.send: addPinnedURL");
        chrome.send('addPinnedURL', [
          data.url,
          data.title,
          data.faviconUrl || '',
          data.thumbnailUrl || '',
          // TODO(estade): should not need to convert index to string.
          String(this.index)
        ]);
      } else {
        console.log(" ---------- chrome.send: removePinnedURL");
        chrome.send('removePinnedURL', [data.url]);
      }
      currentTileIndex = this.index;
      currentTitle = data.title;
      this.updatePinnedState_();
    },

    /**
     * Updates the DOM for the current pinned state.
     */
    updatePinnedState_: function() {
      if (this.data_.pinned) {
        this.classList.add('pinned');
        this.querySelector('.pin').title = loadTimeData.getString('unpinthumbnailtooltip');
      } else {
        this.classList.remove('pinned');
        this.querySelector('.pin').title = loadTimeData.getString('pinthumbnailtooltip');
      }
    },

    /**
     * Permanently removes a page from Most Visited.
     */
    blacklist_: function() {
      chrome.send('blacklistURLFromMostVisited', [this.data_.url]);
      this.reset();
      chrome.send('getMostVisited');
      this.classList.add('blacklisted');
    },

    /**
     * Set the size and position of the most visited tile.
     * @param {number} size The total size of |this|.
     * @param {number} x The x-position.
     * @param {number} y The y-position.
     *     animate.
     */
    setBounds: function(size, x, y) {
      this.style.width = size + 'px';
      this.style.height = heightForWidth(size) + 'px';
      this.style.left = x + 'px';
      this.style.top = y + 'px';
    },
  };

  var mostVisitedPageGridValues = {
    type: 'mosts',
    // The fewest tiles we will show in a row.
    minColCount: 4,
    // The most tiles we will show in a row.
    maxColCount: 4,

    // TODO(estade): Change these to real values.
    // The smallest a tile can be.
    minTileWidth: 201,
    // The biggest a tile can be.
    maxTileWidth: 278
  };
  TilePage.initGridValues(mostVisitedPageGridValues);

  /**
   * Calculates the height for a Most Visited tile for a given width. The size
   * is based on the thumbnail, which should have a 262:154 ratio (the rest of
   * the arithmetic accounts for padding).
   * @return {number} The height.
   */
  function heightForWidth(width) {
    return width * 162 / 278 - 2;
  }

  /**
   * Creates a new MostVisitedPage object.
   * @param {string} name The display name for the page.
   * @constructor
   * @extends {TilePage}
   */
  function MostVisitedPage(name) {
    var el = new TilePage(name, mostVisitedPageGridValues);
    el.__proto__ = MostVisitedPage.prototype;
    el.initialize();

    return el;
  }

  MostVisitedPage.prototype = {
    __proto__: TilePage.prototype,

    initialize: function() {
      this.classList.add('most-visited-page');
      this.data_ = null;
      this.mostVisitedTiles_ = this.getElementsByClassName('most-visited real');
      this.thumbnailCount_ = loadTimeData.getInteger("mostVisitedCount");
      if(loadTimeData.getInteger("thumbOpValue") != localStorage.thumbOpValue)
        localStorage.thumbOpValue = loadTimeData.getInteger("thumbOpValue");
    },
    /**
     * Create blank (filler) tiles.
     * @private
     */
    createTiles_: function() {
      for (var i = 0; i < this.thumbnailCount_; i++) {
        this.appendTile(new MostVisited());
      }  
      var cachedRangeValue = localStorage.thumbOpValue ? localStorage.thumbOpValue : 15; 
      //$("thumbnailOpacityRange").setAttribute("value",cachedRangeValue);
      setThumbOpacity(cachedRangeValue);
    },

    /**
     * Update the tiles after a change to |data_|.
     */
    updateTiles_: function() {
      for (var i = 0; i < this.thumbnailCount_; i++) {
        var page = this.data_[i];
        var tile = this.mostVisitedTiles_[i];
        if (i >= this.data_.length) tile.reset();
        else tile.updateForData(page);
      }
    },

    flushSelf:function (index,uselogo,defaultlogo) {
        this.data_[index].uselogo = uselogo;
        this.data_[index].defaultlogo = defaultlogo;
        var page = this.data_[index];
        var tile = this.mostVisitedTiles_[index];
        if (index >= this.data_.length) 
            tile.reset();
        else 
            tile.updateForData(page);
    },
    emptyTiles_: function() {
      this.emptyTile();
    },
    /**
     * Array of most visited data objects.
     * @type {Array}
     */
    get data() {
      return this.data_;
    },
    set data(data) {
      // The first time data is set, create the tiles.
      if (!this.data_) {
        this.emptyTiles_();
        this.createTiles_();
      } 
      var len = data.length, oldCount = this.thumbnailCount_, old_data = null;
      this.thumbnailCount_ = loadTimeData.getInteger("mostVisitedCount");
      this.data_ = data.slice(0, this.thumbnailCount_);
      var isFiller = oldCount!=this.thumbnailCount_;
      if(isFiller){
        old_data = data;
        this.emptyTiles_();
        this.createTiles_();
        for(var i= 0;i<this.data_.length;i++){
            var item={};
            item.filler = true;
            this.data_[i]=item;
        }
      }
      if(len < this.thumbnailCount_){
        for(var i= len;i<this.thumbnailCount_;i++){
            var item={};
            item.filler = true;
            this.data_[i]=item;
        }
      }
      //console.log("data:",this.data_);
      this.updateTiles_();

      /* add by 360 */
      //
      if (this.thumbnailCount_ == 0) {
        this.tileGrid_.style.height = '1px';
        this.tileGrid_.style.minHeight = '1px';
        this.queueUpdateScrollbars_();
      }else if(isFiller){
        this.data = old_data;
      }

//      if ($("add-dot-container").style.display != "block") {
//        setTimeout(function () {
//            chrome.send("cacheNTP");
//        },200);
//      }
    },

    /** @inheritDoc */
    shouldAcceptDrag: function(dataTransfer) {
      return this.contains(ntp.getCurrentlyDraggingTile());
    },

    /** @inheritDoc */
    heightForWidth: heightForWidth,
  };

  /**
   * We've gotten additional Most Visited data. Update our old data with the
   * new data. The ordering of the new data is not important, except when a
   * page is pinned. Thus we try to minimize re-ordering.
   * @param {Object} oldData The current Most Visited page list.
   * @param {Object} newData The new Most Visited page list.
   * @return The merged page list that should replace the current page list.
   */
  function refreshData(oldData, newData) {
    oldData = oldData.slice(0, THUMBNAIL_COUNT);
    newData = newData.slice(0, THUMBNAIL_COUNT);

    // Copy over pinned sites directly.
    for (var j = 0; j < newData.length; j++) {
      if (newData[j].pinned) {
        oldData[j] = newData[j];
        // Mark the entry as 'updated' so we don't try to update again.
        oldData[j].updated = true;
        // Mark the newData page as 'used' so we don't try to re-use it.
        newData[j].used = true;
      }
    }

    // Look through old pages; if they exist in the newData list, keep them
    // where they are.
    for (var i = 0; i < oldData.length; i++) {
      if (oldData[i].updated) continue;

      for (var j = 0; j < newData.length; j++) {
        if (newData[j].used) continue;

        if (newData[j].url == oldData[i].url) {
          // The background image and other data may have changed.
          oldData[i] = newData[j];
          oldData[i].updated = true;
          newData[j].used = true;
          break;
        }
      }
    }

    // Look through old pages that haven't been updated yet; replace them.
    for (var i = 0; i < oldData.length; i++) {
      if (oldData[i].updated) continue;

      for (var j = 0; j < newData.length; j++) {
        if (newData[j].used) continue;

        oldData[i] = newData[j];
        oldData[i].updated = true;
        newData[j].used = true;
        break;
      }

      if (!oldData[i].updated) oldData[i] = null;
    }

    // Clear 'updated' flags so this function will work next time it's called.
    for (var i = 0; i < THUMBNAIL_COUNT; i++) {
      if (oldData[i]) oldData[i].updated = false;
    }

    return oldData;
  };

  return {
    MostVisitedPage: MostVisitedPage,
    refreshData: refreshData,
  };
});

