﻿// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('options', function() {
  const OptionsPage = options.OptionsPage;
  const ArrayDataModel = cr.ui.ArrayDataModel;

  //
  // BrowserOptions class
  // Encapsulated handling of browser options page.
  //
  function BrowserOptions() {
    OptionsPage.call(this, 'browser',
      templateData.browserPageTabTitle,
      'browserPage');
  }

  cr.addSingletonGetter(BrowserOptions);

  BrowserOptions.prototype = {
    // Inherit BrowserOptions from OptionsPage.
    __proto__: options.OptionsPage.prototype,

    startup_pages_pref_: {
      'name': 'session.urls_to_restore_on_startup',
      'disabled': false
    },

    /**
     * At autocomplete list that can be attached to a text field during editing.
     * @type {HTMLElement}
     * @private
     */
    autocompleteList_: null,

    // The cached value of the instant.confirm_dialog_shown preference.
    //instantConfirmDialogShown_: false,

    /**
     * Initialize BrowserOptions page.
     */
    initializePage: function() {
      // Call base class implementation to start preference initialization.
      OptionsPage.prototype.initializePage.call(this);

      $('safemonLockHomePageButton').onclick = function(event) {
        chrome.send('LaunchSafemonLockHomePage');
      };

      Preferences.getInstance().addEventListener('safemonLockHomePage', function(e) {
        var startupRadios = document.querySelectorAll('[name="startup"], #homepageURL, #startupPageTextarea');
        var val = (e.value || {}).value;
        switch (val) {
          case 0:
            break;
          case 1:
            $$('.homepageURL-safemon').classList.add('show-safemon');
            $('safemonLockHomePageButton').setAttribute('icon', 'lock');
            $('safemonLockHomePageButton').innerText = '主页已锁定';
            $('safemonLockHomePageTip').hidden = false;
            for (var i = 0; i < startupRadios.length; i++) {
              startupRadios[i].setAttribute('disabled', 'disabled');
            }
            break;
          case 2:
            $$('.homepageURL-safemon').classList.add('show-safemon');
            $('safemonLockHomePageButton').setAttribute('icon', 'unlock');
            $('safemonLockHomePageButton').innerText = '锁定主页';
            $('safemonLockHomePageTip').hidden = true;
            for (var i = 0; i < startupRadios.length; i++) {
              startupRadios[i].removeAttribute('disabled');
            }
            break;
        }
      });

      // Wire up controls.
      $('startupUseCurrentButton').onclick = function(event) {
        chrome.send('setStartupPagesToCurrentPages');
      };
      $('defaultSearchManageEnginesButton').onclick = function(event) {
        OptionsPage.navigateToPage('searchEngines');
        chrome.send('coreOptionsUserMetricsAction', ['Options_ManageSearchEngines']);
      };
      $('defaultSearchEngine').onchange = this.setDefaultSearchEngine_;

      document.addEventListener("change", function(e) {
        var targetNode = e.target;
        if (targetNode) {
          if (targetNode.pref == 'session.restore_on_startup' || targetNode.id == 'startupPageTextarea' || targetNode.id == 'homepageURL') {
            chrome.send('sessionPrefChanged', [targetNode.value]);
          }
        }
      });


      var self = this;
      //      $('instantEnabledCheckbox').customChangeHandler = function(event) {
      //        if (this.checked) {
      //          if (self.instantConfirmDialogShown_)
      //            chrome.send('enableInstant');
      //          else
      //            OptionsPage.navigateToPage('instantConfirm');
      //        } else {
      //          chrome.send('disableInstant');
      //        }
      //        return true;
      //      };

      //      $('instantFieldTrialCheckbox').addEventListener('change',
      //          function(event) {
      //            this.checked = true;
      //            chrome.send('disableInstant');
      //          });

      //      Preferences.getInstance().addEventListener('instant.confirm_dialog_shown',
      //          this.onInstantConfirmDialogShownChanged_.bind(this));

      //      Preferences.getInstance().addEventListener('instant.enabled',
      //          this.onInstantEnabledChanged_.bind(this));

      Preferences.getInstance().addEventListener(
        $('homepageUseURLButton').pref,
        this.onHomepageUseNTPChanged_);

      if ($('autoFixSearchCheckbox')) {
        Preferences.getInstance().addEventListener($('autoFixSearchCheckbox').pref,
          function() {
            $('defaultAlterSearchEngine').disabled = !$('autoFixSearchCheckbox').checked;
          });

        $('autoFixSearchCheckbox').customChangeHandler = function(event) {
          $('defaultAlterSearchEngine').disabled = !this.checked;
          chrome.send('enableAutoFixSearch', [String(this.checked)]);
          return true;
        };

        $('defaultAlterSearchEngine').onchange = function() {
          var engineSelect = $('defaultAlterSearchEngine');
          var selectedIndex = engineSelect.selectedIndex;
          if (selectedIndex >= 0) {
            var selection = engineSelect.options[selectedIndex];
            chrome.send('setDefaultAlterSearchEngine', [parseInt(selection.value)]);
          }
        };
      }


      chrome.send('getDefaultAlterSearchEngine');

      chrome.send('fetchPrefs', ["BrowserOptions.setStartupPageTextarea", "session.urls_to_restore_on_startup"]);
      var startupPageTextarea = document.getElementById('startupPageTextarea');
      startupPageTextarea.addEventListener('change', function(event) {
        self.saveStartupPages();
      });

      var homepageField = $('homepageURL');
      homepageField.addEventListener('focus', function(event) {
        self.autocompleteList_.attachToInput(homepageField);
      });
      homepageField.addEventListener('blur', function(event) {
        self.autocompleteList_.detach();
      });
      homepageField.addEventListener('keydown', function(event) {
        // Remove focus when the user hits enter since people expect feedback
        // indicating that they are done editing.
        if (event.keyIdentifier == 'Enter')
          homepageField.blur();
      });

      // Text fields may change widths when the window changes size, so make
      // sure the suggestion list stays in sync.
      window.addEventListener('resize', function() {
        self.autocompleteList_.syncWidthToInput();
      });

      // Ensure that changes are committed when closing the page.
      window.addEventListener('unload', function() {
        if (document.activeElement == homepageField)
          homepageField.blur();
      });
      // add by huangximing 2013-07-22 
      Preferences.getInstance().addEventListener($('IDS_BOSS_KEY_ENABLE').pref, function() {
        $('IDS_BOSS_KEY_MUTE_ENABLE').disabled = !$('IDS_BOSS_KEY_ENABLE').checked;
        $('SettingbossKeyBtn').disabled = !$('IDS_BOSS_KEY_ENABLE').checked;
      });
      // add by huangximing end

      if (!cr.isChromeOS) {
        $('defaultBrowserUseAsDefaultButton').onclick = function(event) {
          chrome.send('becomeDefaultBrowser');
        };

        $('autoLaunch').addEventListener('click',
          this.handleAutoLaunchChanged_);
      }

      var startupPagesList = $('startupPagesList');
      options.browser_options.StartupPageList.decorate(startupPagesList);
      startupPagesList.autoExpands = true;

      // Check if we are in the guest mode.
      if (cr.commandLine && cr.commandLine.options['--bwsi']) {
        // Hide the startup section.
        $('startupSection').hidden = true;
      } else {
        // Initialize control enabled states.
        Preferences.getInstance().addEventListener('session.restore_on_startup',
          this.updateCustomStartupPageControlStates_.bind(this));
        Preferences.getInstance().addEventListener(
          this.startup_pages_pref_.name,
          this.handleStartupPageListChange_.bind(this));

        this.updateCustomStartupPageControlStates_();
      }

      if ($("addNewRulesBtn")) {
        $("addNewRulesBtn").addEventListener("click", function(e) {
          chrome.send("addRule");
        });
      }

      var suggestionList = new cr.ui.AutocompleteList();
      suggestionList.autoExpands = true;
      suggestionList.suggestionUpdateRequestCallback =
        this.requestAutocompleteSuggestions_.bind(this);
      $('main-content').appendChild(suggestionList);
      this.autocompleteList_ = suggestionList;
      startupPagesList.autocompleteList = suggestionList;
    },

    onGetDefaultAlterSearchEngine_: function(value) {
      if ($('defaultAlterSearchEngine') && (value == '0' || value == '1' || value == '2')) {
        $('defaultAlterSearchEngine').value = value;
      }
    },

    saveStartupPages: function() {
      var startupPageTextarea = document.getElementById('startupPageTextarea');
      var urls = startupPageTextarea.value.split('\n'),
        len = urls.length,
        urlsArr = [],
        checkUrl = function(url) {
          url = url.replace(/(^\s*)|(\s*$)/g, "");
          if (url != '') {
            urlsArr.push(url);
          }
        };
      for (var i = 0; i < len; ++i) {
        checkUrl(urls[i]);
      }
      Preferences.setListPref(this.startup_pages_pref_.name, urls);
    },


    /**
     * Called when the value of the instant.confirm_dialog_shown preference
     * changes. Cache this value.
     * @param {Event} event Change event.
     * @private
     */
    //    onInstantConfirmDialogShownChanged_: function(event) {
    //      this.instantConfirmDialogShown_ = event.value['value'];
    //    },

    /**
     * Called when the value of the instant.enabled preference changes. Request
     * the state of the Instant field trial experiment.
     * @param {Event} event Change event.
     * @private
     */
    //    onInstantEnabledChanged_: function(event) {
    //      chrome.send('getInstantFieldTrialStatus');
    //    },

    /**
     * Called to set the Instant field trial status.
     * @param {boolean} enabled If true, the experiment is enabled.
     * @private
     */
    //    setInstantFieldTrialStatus_: function(enabled) {
    //      $('instantEnabledCheckbox').hidden = enabled;
    //      $('instantFieldTrialCheckbox').hidden = !enabled;
    //      $('instantLabel').htmlFor = enabled ? 'instantFieldTrialCheckbox'
    //                                          : 'instantEnabledCheckbox';
    //    },

    /**
     * Called when the value of the homepage-use-NTP pref changes.
     * Updates the disabled state of the homepage text field.
     * Notice that the text field can be disabled for other reasons too
     * (it can be managed by policy, for instance).
     * @param {Event} event Change event.
     * @private
     */
    onHomepageUseNTPChanged_: function(event) {
      var homepageField = $('homepageURL');
      var homepageUseURLButton = $('homepageUseURLButton');
      homepageField.setDisabled('radioNotSelected', !homepageUseURLButton.checked);
      $('safemonLockHomePageButton').disabled = !homepageUseURLButton.checked;
    },

    /**
     * Update the Default Browsers section based on the current state.
     * @param {string} statusString Description of the current default state.
     * @param {boolean} isDefault Whether or not the browser is currently
     *     default.
     * @param {boolean} canBeDefault Whether or not the browser can be default.
     * @private
     */
    updateDefaultBrowserState_: function(statusString, isDefault,
      canBeDefault) {
      var label = $('defaultBrowserState');
      label.textContent = statusString;

      $('defaultBrowserUseAsDefaultButton').disabled = !canBeDefault ||
        isDefault;
    },

    /**
     * Clears the search engine popup.
     * @private
     */
    clearSearchEngines_: function() {
      $('defaultSearchEngine').textContent = '';
    },

    /**
     * Updates the search engine popup with the given entries.
     * @param {Array} engines List of available search engines.
     * @param {number} defaultValue The value of the current default engine.
     * @param {boolean} defaultManaged Whether the default search provider is
     *     managed. If true, the default search provider can't be changed.
     */
    updateSearchEngines_: function(engines, defaultValue, defaultManaged) {
      this.clearSearchEngines_();
      engineSelect = $('defaultSearchEngine');
      engineSelect.disabled = defaultManaged;
      engineCount = engines.length;
      var defaultIndex = -1;
      for (var i = 0; i < engineCount; i++) {
        var engine = engines[i];
        var option = new Option(engine['name'], engine['index']);
        if (defaultValue == option.value)
          defaultIndex = i;
        engineSelect.appendChild(option);
      }
      if (defaultIndex >= 0)
        engineSelect.selectedIndex = defaultIndex;
    },

    /**
     * Returns true if the custom startup page control block should
     * be enabled.
     * @returns {boolean} Whether the startup page controls should be
     *     enabled.
     */
    shouldEnableCustomStartupPageControls: function(pages) {
      return $('startupShowPagesButton').checked &&
        !this.startup_pages_pref_.disabled;
    },

    /**
     * Updates the startup pages list with the given entries.
     * @param {Array} pages List of startup pages.
     * @private
     */
    updateStartupPages_: function(pages) {
      var model = new ArrayDataModel(pages);
      // Add a "new page" row.
      model.push({
        'modelIndex': '-1'
      });
      $('startupPagesList').dataModel = model;
    },

    /**
     * Called when the default content setting value for a content type changes.
     * @param {Object} dict A mapping content setting types to the default
     * value.
     * @private
     */
    setContentFilterSettingsValue_: function(dict) {
      if ('cookies' in dict && 'value' in dict['cookies'])
        this.sessionOnlyCookies_ = dict['cookies']['value'] == 'session';
    },

    /**
     * Sets the enabled state of the custom startup page list controls
     * based on the current startup radio button selection.
     * @private
     */
    updateCustomStartupPageControlStates_: function() {
      var disable = !this.shouldEnableCustomStartupPageControls();
      $('startupPageTextarea').disabled = disable;
      var startupPagesList = $('startupPagesList');
      startupPagesList.disabled = disable;
      startupPagesList.setAttribute('tabindex', disable ? -1 : 0);
      // Explicitly set disabled state for input text elements.
      var inputs = startupPagesList.querySelectorAll("input[type='text']");
      for (var i = 0; i < inputs.length; i++)
        inputs[i].disabled = disable;
      $('startupUseCurrentButton').disabled = disable;
    },

    /**
     * Handles change events of the preference
     * 'session.urls_to_restore_on_startup'.
     * @param {event} preference changed event.
     * @private
     */
    handleStartupPageListChange_: function(event) {
      this.startup_pages_pref_.disabled = event.value['disabled'];
      this.updateCustomStartupPageControlStates_();
    },

    /**
     * Sets the default search engine based on the popup selection.
     */
    setDefaultSearchEngine_: function() {
      var engineSelect = $('defaultSearchEngine');
      var selectedIndex = engineSelect.selectedIndex;
      if (selectedIndex >= 0) {
        var selection = engineSelect.options[selectedIndex];
        chrome.send('setDefaultSearchEngine', [String(selection.value)]);
      }
    },

    /**
     * Sets or clear whether Chrome should Auto-launch on computer startup.
     */
    handleAutoLaunchChanged_: function() {
      chrome.send('toggleAutoLaunch', [Boolean($('autoLaunch').checked)]);
    },

    /**
     * Sends an asynchronous request for new autocompletion suggestions for the
     * the given query. When new suggestions are available, the C++ handler will
     * call updateAutocompleteSuggestions_.
     * @param {string} query List of autocomplete suggestions.
     * @private
     */
    requestAutocompleteSuggestions_: function(query) {
      chrome.send('requestAutocompleteSuggestions', [query]);
    },

    /**
     * Updates the autocomplete suggestion list with the given entries.
     * @param {Array} pages List of autocomplete suggestions.
     * @private
     */
    updateAutocompleteSuggestions_: function(suggestions) {
      var list = this.autocompleteList_;
      // If the trigger for this update was a value being selected from the
      // current list, do nothing.
      if (list.targetInput && list.selectedItem &&
        list.selectedItem['url'] == list.targetInput.value)
        return;
      list.suggestions = suggestions;
    },

    /**
     * Shows the autoLaunch preference and initializes its checkbox value.
     */
    updateAutoLaunchState_: function(enabled) {
      $('autoLaunchOption').hidden = false;
      $('autoLaunch').checked = enabled;
    },
  };

  BrowserOptions.updateDefaultBrowserState = function(statusString, isDefault,
    canBeDefault) {
    if (!cr.isChromeOS) {
      BrowserOptions.getInstance().updateDefaultBrowserState_(statusString,
        isDefault,
        canBeDefault);
    }
  };

  BrowserOptions.updateSearchEngines = function(engines, defaultValue,
    defaultManaged) {
    BrowserOptions.getInstance().updateSearchEngines_(engines, defaultValue,
      defaultManaged);
  };

  BrowserOptions.updateStartupPages = function(pages) {
    BrowserOptions.getInstance().updateStartupPages_(pages);
  };

  BrowserOptions.setContentFilterSettingsValue = function(dict) {
    BrowserOptions.getInstance().setContentFilterSettingsValue_(dict);
  };

  BrowserOptions.updateAutocompleteSuggestions = function(suggestions) {
    BrowserOptions.getInstance().updateAutocompleteSuggestions_(suggestions);
  };

  BrowserOptions.updateAutoLaunchState = function(enabled) {
    BrowserOptions.getInstance().updateAutoLaunchState_(enabled);
  };

  BrowserOptions.setInstantFieldTrialStatus = function(enabled) {
    //BrowserOptions.getInstance().setInstantFieldTrialStatus_(enabled);
  };

  BrowserOptions.onGetDefaultAlterSearchEngine = function(value) {
    BrowserOptions.getInstance().onGetDefaultAlterSearchEngine_(value);
  };

  BrowserOptions.updateRulesList = function(ruleList) {
    if (!ruleList) return;
    var ruleLen = ruleList.length;
    var ruleDiv = $("ruleListContainer");
    var by = function(name) {
      return function(o, p) {
        var a, b;
        if (typeof o === "object" && typeof p === "object" && o && p) {
          a = o[name];
          b = p[name];
          if (a === b) {
            return 0;
          }
          if (typeof a === typeof b) {
            return a < b ? -1 : 1;
          }
          return typeof a < typeof b ? -1 : 1;
        } else {
          throw ("error");
        }
      }
    };
    ruleList.sort(by("id"));
    var list = document.querySelectorAll(".rulesList");
    for (var j = 0; j < list.length; j++) {
      list[j].parentNode.removeChild(list[j]);
    }
    for (var i = 0; i < ruleLen; i++) {
      var rowDiv = document.createElement("div");
      rowDiv.className = "row rulesList";

      var colOne = document.createElement("div");
      colOne.className = "one";

      var enableInput = document.createElement("input");
      enableInput.type = "checkbox";
      enableInput.value = ruleList[i].id;
      if (ruleList[i].id < 0)
        enableInput.setAttribute("disabled", "true");

      enableInput.checked = ruleList[i].isEnable ? true : false;
      enableInput.addEventListener("click", function(e) {
        chrome.send("enableRule", [parseInt(this.value), Boolean(this.checked)]);
      });

      colOne.appendChild(enableInput);

      var colTwo = document.createElement("div");
      colTwo.className = "two";
      colTwo.innerHTML = ruleList[i].title;

      var colThree = document.createElement("div");
      colThree.className = "three";

      var updateTimeCon = document.createElement("input");
      updateTimeCon.type = "text";
      updateTimeCon.value = ruleList[i].lastTime;
      updateTimeCon.setAttribute("readonly", "true");
      colThree.appendChild(updateTimeCon);

      var colFour = document.createElement("div");
      colFour.className = "three";

      if (ruleList[i].id >= 0) {
        var replaceRule = document.createElement("a");
        replaceRule.href = "javascript:void(0)";
        replaceRule.innerText = templateData.replaceRules;
        replaceRule.onclick = (function(ruleId) {
          var ruleid = ruleId;
          return function() {
            chrome.send("editRule", [parseInt(ruleid)]);
          };
        })(ruleList[i].id);

        var deleteRule = document.createElement("a");
        deleteRule.href = "javascript:void(0)";
        deleteRule.innerText = templateData.deleteRules;
        deleteRule.onclick = (function(ruleId) {
          var ruleid = ruleId;
          return function() {
            chrome.send("deleteRule", [parseInt(ruleid)]);
          };
        })(ruleList[i].id);

        colFour.appendChild(replaceRule);
        colFour.appendChild(deleteRule);
      }
      rowDiv.appendChild(colOne);
      rowDiv.appendChild(colTwo);
      rowDiv.appendChild(colThree);
      rowDiv.appendChild(colFour);

      ruleDiv.appendChild(rowDiv);
    }
    //todo
  };
  BrowserOptions.disableAutoFixed = function() {
    if ($("autoFixCon")) $("autoFixCon").style.display = "none";
  };
  BrowserOptions.setStartupPageTextarea = function(dict) {
    var startupPageTextarea = document.getElementById('startupPageTextarea');
    var urls = dict.session.urls_to_restore_on_startup.value,
      len = urls.length,
      text = '';
    for (var i = 0; i < len; ++i) {
      text += ((i == 0 ? '' : '\n') + urls[i]);
    }
    startupPageTextarea.innerText = text;
  };
  // Export
  return {
    BrowserOptions: BrowserOptions
  };

});