//
//  GSGlyphsInfo.h
//  Glyphs
//
//  Created by Georg Seifert on 20.1.08.
//  Copyright 2008 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class GSGlyphInfo;
@class GSGlyph;
@class GSFont;
@class GSLayer;
@class MGOrderedDictionary;

/*! \brief GSGlyphsInfos: This is a singleton class to provide glyphinfo services.
 \ingroup common
 This is a singleton class to provide glyphinfo services.
 */
@interface GSGlyphsInfo : NSObject {
	NSURL *_localGlyphInfoURL;
	NSMutableArray* _glyphInfos;
	NSMutableArray *_categories;
	NSMutableArray *_subCategories;
	NSArray* _glyphGroupsTree;
	NSMutableArray* _sidebarCategories;
	NSDictionary* _categoryIndexes;
	NSDictionary* _sidebarSubcategories;
	NSMutableDictionary* _name2Group;
	NSMutableDictionary* _name2Info;
	NSMutableDictionary* _unicode2Info;
	NSDictionary* _languages;
	NSMutableArray* _scripts;
	NSDictionary* _scriptIndexes;
	NSDictionary* _anchors;
	NSDictionary* _customParameterTypes;
	NSArray* _customFontParameters;
	NSArray* _customMasterParameters;
	NSArray* _customInstanceParameters;
	NSDictionary* _basicFontInfoProperties;
	IBOutlet NSArrayController* _glyphGroupsTreeTreeController;
	bool updating;

	NSDictionary* _widthClasses;
	NSDictionary* _weightClasses;
	NSDictionary* _scriptAbbreviations;
	NSDictionary* _scriptSuffixes;
	NSDictionary* _languageScripts;
	NSArray* _languageData;
	NSArray* _unicodeRanges;
	NSArray* _codePageRanges;
	NSMutableDictionary* _scriptRanges;
	NSSet* _indicScripts;
	NSSet* _rtlScripts;
@private
	NSMutableDictionary* _subGroupCache;
	NSDateFormatter* _dateFormat;
	NSMutableDictionary *_macLangIDs;
	NSMutableDictionary *_macEncodingIDs;
	NSMutableDictionary *_winLangIDs;

}
/** An array of glyph groups.

This is used the build the filter tree in font view mode.
*/
@property (readonly, nonatomic) NSArray* glyphGroupsTree;
@property (readonly, nonatomic) NSMutableArray* sidebarCategories;
@property (readonly, nonatomic) NSMutableDictionary* name2Group;
/// The array containing the glyphs info data.
@property (strong, nonatomic) NSMutableArray* glyphInfos;
@property (readonly, nonatomic) NSMutableArray* scripts;
@property (nonatomic, readonly) NSArray* categories;
@property (nonatomic, readonly) NSArray* subCategories;
@property (readonly, nonatomic) NSDictionary* customParameterTypes;
@property (readonly, nonatomic) NSArray* customFontParameters;
@property (readonly, nonatomic) NSArray* customMasterParameters;
@property (readonly, nonatomic) NSArray* customInstanceParameters;

@property (readonly, nonatomic) NSDictionary* basicFontInfoProperties;
@property (readonly, nonatomic) NSDictionary* languages;
/// The width classes
@property (readonly, nonatomic) NSDictionary* widthClasses;
/// The weight classes
@property (readonly, nonatomic) NSDictionary* weightClasses;
@property (readonly, nonatomic) NSDictionary* name2Info;
@property (readonly, nonatomic) NSDictionary* scriptRanges;
@property (readonly, nonatomic) NSSet* indicScripts;
@property (readonly, nonatomic) NSSet* rtlScripts;
@property (readonly, nonatomic) NSDictionary* iMatraCombinations;

/// Returns the shared GlyphsInfos object.
+ (GSGlyphsInfo*)sharedManager;

- (instancetype)initWithLocalFile:(NSURL *)localGlyphInfoURL;

- (void)loadGlyphInfo;

+ (NSDateFormatter*)dateFormat;

+ (NSString*)applicationSupportFolder;

- (void)reloadCustomFilterGroup;

- (NSDictionary*)loadCustomFilterGroup;

/** Returns the GSGlyphInfo for the glyph with Glyph.

@param Glyph The glyph.
@return A GSGlyphInfo instance or nil.
*/
- (GSGlyphInfo*)glyphInfoForGlyph:(GSGlyph*)Glyph;

/** Returns the GSGlyphInfo for the glyph with Name.

@param Name The glyph name.
@return A GSGlyphInfo instance or nil.
*/
- (GSGlyphInfo*)glyphInfoForName:(NSString*)Name;

/** Returns the GSGlyphInfo for the glyph with Unicode.

@param Unicode A unicode as Hex string.
@return A GSGlyphInfo instance or nil.
*/
- (GSGlyphInfo*)glyphInfoForUnicode:(NSString*)Unicode;

- (GSGlyphInfo*)glyphInfoForName:(NSString*)Name componentNames:(NSArray*)ComponentNames LangTag:(NSString*)LangTag suffixes:(NSArray *)cutSuffixes;

- (GSGlyphInfo*)glyphInfo:(GSGlyph*)Glyph changeName:(BOOL)ChangeName thin:(BOOL)Thin;

+ (void)applyGlyphInfo:(GSGlyphInfo*)GlyphInfo toGlyph:(GSGlyph*)Glyph changeName:(BOOL)ChangeName;

/// Returns the index of the glyph in the database.
- (NSUInteger)glyphIndex:(GSGlyph*)Glyph;

+ (BOOL)isRLTScript:(NSString*)script;
/** returns the name of the lowercase glyph name linked to the given uppercase name
 
 in most instances this is equivalent with `[Name lowercaseString]` but there are some exceptions. e.g. A > a, KoppaArchaic > koppaArchaic
 
 @param Name a upper case glyph name
 
 @return the matching lower case name
 */
- (NSString *)lowerCaseName:(NSString *)Name;

/**
 Updates the glyph info in the Glyph.
 
 @param glyph      The Glyph
 @param changeName Keep the name and only update unicode, categories and script
 */
- (void)updateGlyphInfo:(GSGlyph*)glyph changeName:(BOOL)changeName;
#ifndef GLYPHS_VIEWER

///

/** Generates missing anchors in the Layer.
 
 @param Layer The Layer
 
 @return YES if it changed anything.
 */
- (BOOL)updateAnchor:(GSLayer*)Layer;

#endif

/** Converts a name to a standardised name

 e.g. converts afii10017 to A-cy
 
 @param name The name.
 @return The nice-name.
 
 @see productionGlyphNameForName:
*/
- (NSString*)niceGlyphNameForName:(NSString*)name;

/** Converts a name to a production name
 
 @param name The name.
 @return The production name.
 @see niceGlyphNameForName:
 */
- (NSString*)productionGlyphNameForName:(NSString*)name;

- (NSString*)productionGlyphNameForGlyph:(GSGlyph*)glyph;

/** Converts all glyph names to production names 
 
 This is useful for OT-feature code
 
 @param string A string that contains glyph names like feature code.
 @return The string with converted names.
 */
- (NSString*)convertToLegecyNames:(NSString*)string;

+ (NSString *)convertNames:(NSDictionary *)changeNames inString:(NSString *)FeaturesString;

- (void)fixASCIINames:(GSFont*)Font;

- (NSString*)decomposeHangul:(unichar)s;

+ (BOOL)decomposeHangul:(unichar)s leadingConsonant:(unichar*)LeadingConsonant vowel:(unichar*)Vowel trailingConsonant:(unichar*)TrailingConsonant;

+ (unichar)composeHangulLeadingConsonant:(unichar)LeadingConsonant vowel:(unichar)Vowel trailingConsonant:(unichar)TrailingConsonant;

- (short)hangulComponentPositionForName:(NSString*)name;

- (short)hangulComponentPositionForUnichar:(unichar)Char;

- (NSArray*)hangulJamoGlyphsForPos:(short)pos;

+ (NSString*)hangulPositionKeyForName:(NSString*)name;

+ (NSArray*) hangulFirstConsonants;
+ (NSArray*) hangulVowels;
+ (NSArray*) hangulFinalConsonants;

- (NSArray*)compleationsForName:(NSString*)PartialName forPartialWordRange:(NSRange)charRange;

- (NSUInteger)indexOfObjectInScripts:(NSString*)Script;

- (NSUInteger)_indexOfObjectInSidebarCategories:(NSString*)Category;

- (NSMutableDictionary*)_subcategoriesForSidebarCategory:(NSString*)Category;

+ (NSDictionary*)scriptAbbreviations;

+ (NSDictionary*)scriptSuffixes;

+ (NSDictionary*)languageScripts;

+ (NSArray*)languageData;

- (NSNumber *)macEncodingIDForLanguage:(NSString *)language;
- (NSNumber *)macLanguageIDForLanguage:(NSString *)language;
- (NSNumber *)winLanguageIDForLanguage:(NSString *)language;

- (NSNumber *)winLanguageForLanguageID:(NSInteger)languageID;

+ (NSArray*)unicodeRanges;

+ (NSArray *)codePageRanges;

- (NSString*)ligaNameForComponents:(NSArray*)Componetents;

- (NSArray*)_componentsForLigaName:(NSString*)LigaName;

+ (NSString*)_langTagForName:(NSString*)Name;

+ (NSArray*)blueValues:(NSArray*)AlignmentZones;

+ (NSArray*)otherBlues:(NSArray*)AlignmentZones;

@end
