//
//  GSInstance.h
//  Glyphs
//
//  Created by Georg Seifert on 5.2.08.
//  Copyright 2008 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class GSFont;
@class GSCustomParameter;
#ifndef GLYPHS_LITE
@class GSInterpolationFontProxy;
#endif

#import <GlyphsCore/GSContainerProtocol.h>

/** The class defining the instance object
*/

@interface GSInstance : NSObject <NSCoding, NSCopying, GSContainerProtocol> {
	GSFont* __unsafe_unretained _font;
	NSString* _name;
	NSString* _linkStyle;
	float _interpolationWeight;
	float _interpolationWidth;
	float _interpolationCustom;
	float _interpolationCustom1;
	float _interpolationCustom2;
	NSString* _weightClass;
	NSString* _widthClass;
	BOOL _exports;
	BOOL _visible;
	BOOL _isItalic;
	BOOL _isBold;
	NSMutableDictionary* _instanceInterpolations;
	NSMutableDictionary* _tempData;
	NSMutableArray* _customParameters;
#ifndef GLYPHS_LITE
	BOOL _manualInterpolation;
	GSInterpolationFontProxy *_interpolatedFont;
#endif
}

/** reference to the containing font */
@property (unsafe_unretained, nonatomic) GSFont* font;

/** @name info */

/** The Intances name

 This is used as the style name.
 */
@property (copy, nonatomic) NSString* name;

/** The style to use as a the regular for this style.

 if styleName is `Medium`, you could link to `Light`. So this instance becomes the Bold stil for the Light.
 */
@property (copy, nonatomic) NSString* linkStyle;

/// The interpolation Weight position.
@property (nonatomic) float interpolationWeight;

/// The interpolation Width position
@property (nonatomic) float interpolationWidth;

/// The interpolation Cutom position
@property (nonatomic) float interpolationCustom;
/// The interpolation Cutom position
@property (nonatomic) float interpolationCustom1;
/// The interpolation Cutom position
@property (nonatomic) float interpolationCustom2;
#ifndef GLYPHS_LITE
@property (nonatomic, readonly) GSInterpolationFontProxy* interpolatedFont;
#endif

/** Weight Class

 e.g. Regular or Bold
 */
@property (strong, nonatomic) NSString* weightClass;

/// returnes the actual value as set by the weightClass or custom parameter "weightClass"
@property (nonatomic) NSUInteger weightClassValue;

/** Width Class

 e.g. Medium (normal)
 */
@property (strong, nonatomic) NSString* widthClass;

/// returnes the actual value as set by the widthClass
@property (nonatomic) NSUInteger widthClassValue;

/// If it should be exported.
@property (nonatomic) BOOL exports;

/// If it should be visible in preview.
@property (nonatomic) BOOL visible;

/// the Style linking Italic bit
@property (nonatomic) BOOL isItalic;

/// the Style linking Bold bit
@property (nonatomic) BOOL isBold;

/// The content of the instance to store it in a pList.
@property (unsafe_unretained) NSDictionary* instanceDict;

/** To store stuff temporarily
 
 This is not saved to file
 */
@property (strong, nonatomic) NSMutableDictionary* tempData;

- (instancetype)initWithInstance:(GSInstance*)Instance;

- (BOOL)isEqualToInstance:(GSInstance *)object;
#ifndef GLYPHS_VIEWER

- (BOOL)saveToFile:(FILE*)File error:(NSError *__autoreleasing*)error;

/** A dict that contains the interpolation coefficents for each master.
 
 This is automatcially updated if you change interpolationWeight, interpolationWidth, interpolationCustom. It contains FontMaster IDs as keys and coeffients for that master as values.
 
 Or, you can set it maually if you set manualInterpolation to true. There is no UI for this, so you need to do that with a script
 */
#ifndef LIBCORE
@property (strong, nonatomic) NSDictionary* instanceInterpolations;
#else
@property (strong, nonatomic) NSMutableDictionary* instanceInterpolations;
#endif

/** Disables automatic calculation of instanceInterpolations
 
 This allowes to manually setting instanceInterpolations.
 */
@property (nonatomic) BOOL manualInterpolation;

/** Calculates the coefficients for the given masters
 
 This is used to automatically calculate the instanceInterpolations
 
 @param masters A list of GSFontMaster objects
 
 @return a dict of FontMaster IDs as keys and coeffients as values.
 */
//- (NSDictionary *)instanceInterpolations:(NSArray *)masters;

- (NSDictionary *)instanceInterpolationsWithUpdates:(BOOL)UpdateKVO;

/** Call it if you have made changes to the instance to force an update of instanceInterpolations. */
- (void) updateInterpolationValues ;

- (NSString *)instanceKey;

/** @name customParameters */

/// A list of GSCustomParameter objects
@property (strong, nonatomic) NSMutableArray* customParameters;

/** The count of custom parameters */
- (NSUInteger)countOfCustomParameters;

- (GSCustomParameter*)objectInCustomParametersAtIndex:(NSUInteger)theIndex;

/** The value of the custom parameter where name == Key
 
 @param Key The name of the paremter
 
 @return the value of the first matching parameter
 */
- (id)customValueForKey:(NSString*)Key;

/** The custom parameter where name == Key
 
 @param Key The name of the paremter
 
 @return the first matching paramter
 */
- (GSCustomParameter*)customParameterForKey:(NSString*)Key;

/** Set the value for key
 
 It will look for an existing paramter with the name an overwrite its value, or adds a new parameter
 @param value The value to add
 @param key   The name of the paramter
 */
- (void)setCustomParameter:(id)value forKey:(NSString*)key;

/** Removes the first paramter with then Name
 
 @param Key The name
 */
- (void)removeObjectFromCustomParametersForKey:(NSString*)Key;

- (void)insertObject:(GSCustomParameter*)aCustomProperty inCustomParametersAtIndex:(NSUInteger)idx;

- (void)removeObjectFromCustomParametersAtIndex:(NSUInteger)idx;

- (void)replaceObjectInCustomParametersAtIndex:(NSUInteger)idx withObject:(GSCustomParameter*)aCustomProperty;

/** Scales the instance object.
 
 @param Scale Scale value
 */
- (void)scaleBy:(CGFloat)Scale;
#endif
- (NSString *)fileName;
- (NSString *)fileName:(int)format;
- (NSString *)preferredFamily;
- (NSString *)preferredSubfamilyName;
- (NSString *)macFullName;
- (NSString *)macStyle;
- (NSString *)windowsFamily;
- (NSString *)windowsStyle;
- (NSString *)fontName;
- (NSString *)fullName;
@end
