//
//  GlyphsFilterProtocol.h
//  GlyphsCore
//
//  Created by Georg Seifert on 08.05.09.
//  Copyright 2009 schriftgestaltung.de. All rights reserved.
//

@class GSLayer;
@class GSFont;
@class GSDialogController;
@class GSInstance;
@class GSGlyph;

/** The protocol defining the filter plugin interface */
@protocol GlyphsFilter
/** The interface version.*/
@property (nonatomic, readonly) NSUInteger interfaceVersion;
/** The title of the plugin.*/
@property (nonatomic, readonly) NSString* title;
/** The keyEquivalent to select the Filter.*/
@property (nonatomic, readonly) NSString* keyEquivalent;
/** the Controller of type GSEditViewController */
@property (nonatomic, unsafe_unretained) id controller;


/// Glyphs will call that before the filter is invoked.
- (NSError*)setup;

/// This method will be called if one layer is active in Edit View (the nodes are visible) Pay attention to the selection.
- (BOOL)runFilterWithLayer:(GSLayer*)Layer error:(out NSError *__autoreleasing*)error;

/** This method will be called if one layer is active in Edit View
 
 @param Layer   The active Layer
 @param options A dict containing some options. This might be "CheckSelection" : YES/NO.
 @param error   if something goes wrong, retunes an NSError by reference.
 
 @return YES if successful, otherwise NO. In the later case the error will hold a explanation.
 */
- (BOOL)runFilterWithLayer:(GSLayer*)Layer options:(NSDictionary *)options error:(out NSError *__autoreleasing*)error;

/// This method will be called if the hole glyph is supposed to be processed.
- (BOOL)runFilterWithLayers:(NSArray*)Layers error:(out NSError *__autoreleasing*)error;

@optional

//! do main initialisations
- (void) loadPlugin;

@property (nonatomic, readonly) NSView* view;

/// is set, if the user chose "run last Filter". Then it should keep the settings from the last run.
@property (nonatomic) BOOL rerun;

/// the returned value will be placed as the title of the "OK" button in the filter dialog.
- (NSString*)actionName;

/** @name Export Filter Support */

/// This will be call if the filter should be run from on export of the font. (see: Instance > Custom Properties)
- (BOOL)filterNeedsRemoveOverlap;

/** Is called to process font when called from a 'Filter' custom parameter
 
 @param Font      The font to process
 @param Arguments A list of arguments
 */
- (void)processFont:(GSFont*)Font withArguments:(NSArray*)Arguments;

/** Is called to process a layer when to preview a Filter custom parameter
 
 @param Layer     The layer to process
 @param Arguments A list of arguments
 */
- (void)processLayer:(GSLayer*)Layer withArguments:(NSArray*)Arguments;

/// Implement this to enable the
- (NSString*)customParameterString;
/** Implement it if you need to suply your own window.
 
 @return a GSDialogController that controlles the window
 */
- (GSDialogController *)makeWindowController;
@end

@protocol GlyphsCallback <NSObject>

- (NSUInteger)callOrder;
/** Is called to process a layer before output
 
 @param layer     The layer to process
 @param instance  The instance that is used to interpolate
 */
- (void)processLayer:(GSLayer*)layer originalGlyph:(GSGlyph*)originalGlyph withInstance:(GSInstance*)instance;

/** Is called to process a bezierpath that belongs to the layer
 
 @param bezierPath     The bezierpath to process
 @param openBezierPath     The openBezierPath to process
 @param secondaryPath The secondaryPath
 @param layer     The layer
 */

- (void)processBezierPath:(NSBezierPath*)bezierPath openBezierPath:(NSBezierPath*)openBezierPath secondaryPath:(NSBezierPath*)secondaryPath forLayer:(GSLayer*)layer;

@end
