/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.gdi;

import com.jniwrapper.Int;
import com.jniwrapper.Parameter;
import com.jniwrapper.Structure;
import com.jniwrapper.UInt32;

/**
 * This class represents ICONINFO structure.
 *
 * @author Serge Piletsky
 */
public class IconInfo extends Structure
{
    private Int _icon = new Int();
    private UInt32 _xHotSpot = new UInt32();
    private UInt32 _yHotSpot = new UInt32();
    private Bitmap _maskBitmap;
    private Bitmap _colorBitmap;

    /**
     * Creates instance of the <code>IconInfo</code>.
     */
    public IconInfo()
    {
        _maskBitmap = new DDBitmap();
        _colorBitmap = new DDBitmap();
        init(new Parameter[]{_icon, _xHotSpot, _yHotSpot, _maskBitmap, _colorBitmap, }, (short)8);
    }

    /**
     * Creates instance of the <code>IconInfo</code>.
     * 
     * @param maskBitmap The icon bitmask bitmap. If this structure defines a black and white icon,
     * this bitmask is formatted so that the upper half is the icon AND bitmask and the lower half is the
     * icon XOR bitmask. Under this condition, the height should be an even multiple of two.
     * If this structure defines a color icon, this mask only defines the AND bitmask of the icon.
     * @param colorBitmap The icon color bitmap.
     */
    public IconInfo(Bitmap maskBitmap, Bitmap colorBitmap)
    {
        _maskBitmap = maskBitmap;
        _colorBitmap = colorBitmap;
        init(new Parameter[]{_icon, _xHotSpot, _yHotSpot, _maskBitmap, _colorBitmap, }, (short)8);
    }

    /**
     * Creates instance of the <code>IconInfo</code> and initializes it by the specified one.
     *
     * @param that Instance of the <code>IconInfo</code>.
     */
    public IconInfo(IconInfo that)
    {
        this();
        initFrom(that);
    }

    /**
     * Returns the icon bitmask bitmap.
     *  
     * @return The icon bitmask bitmap
     */
    public Bitmap getMaskBitmap()
    {
        return _maskBitmap;
    }

    /**
     * Returns handle to the icon color bitmap.
     *  
     * @return Handle to the icon color bitmap.
     */
    public Bitmap getColorBitmap()
    {
        return _colorBitmap;
    }

    /**
     * Specifies whether this structure defines an icon.
     *
     * @return  true if this structure defines an icon. Otherwise false.
     */
    public boolean isIcon()
    {
        return _icon.getValue() != 0;
    }

    /**
     * Specifies whether this structure defines a cursor.
     *
     * @return true if this structure defines an cursor. Otherwise false. 
     */
    public boolean isCursor()
    {
        return !isIcon();
    }

    /**
     * Returns the x-coordinate of the hot spot of the cursor.
     * If this structure defines an icon, the hot spot is always in the center of the icon,
     * and this member is ignored.
     *
     * @return The x-coordinate of the hot spot of the cursor.
     */
    public UInt32 getXHotSpot()
    {
        return _xHotSpot;
    }

    /**
     * Returns the y-coordinate of the hot spot of the cursor.
     * If this structure defines an icon, the hot spot is always in the center of the icon,
     * and this member is ignored.
     * 
     * @return The y-coordinate of the hot spot of the cursor.
     */
    public UInt32 getYHotSpot()
    {
        return _yHotSpot;
    }

    /**
     * Creates copy of this instance
     *
     * @return Copy of this instance
     */
    public Object clone()
    {
        return new IconInfo(this);
    }
}