/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.hook;

import com.jniwrapper.win32.ui.Wnd;

/**
 * This class describes events of the {@link
 * com.jniwrapper.win32.hook.Hook.Descriptor#JOURNALRECORD} hook. This event
 * contains a message to be recorded.
 * 
 * @author Serge Piletsky
 */
public class JournalRecordEvent extends HookEventObject
{
    private static final int HC_SYSMODALON = 4;
    private static final int HC_SYSMODALOFF = 5;

    private long _code;
    private long _message;
    private long _paramL;
    private long _paramH;
    private long _time;
    private Wnd _wnd;

    /**
     * Creates instance of the <code>JournalRecordEvent</code> that contains information about a hardware
     *  message sent to the system message queue.
     * 
     * @param source The object on which the event initially occurred.
     * @param code Value that specifies how to process the message.
     */
    public JournalRecordEvent(Object source, long code)
    {
        super(source);
        _code = code;
    }

    /**
     * Creates instance of the <code>JournalRecordEvent</code> that contains full information about a hardware
     *  message sent to the system message queue.
     *
     * @param source The object on which the event initially occurred.
     * @param code Value that specifies how to process the message.
     * @param message Value that specifies the message.
     * @param paramL Value that specifies additional information about the message. The exact meaning depends on the <code>message</code> value.
     * @param paramH Value that specifies additional information about the message. The exact meaning depends on the <code>message</code> value.
     * @param time Value that specifies the time at which the message was posted.
     * @param wnd Handle to the window to which the message was posted.
     */
    public JournalRecordEvent(Object source, long code, long message, long paramL, long paramH, long time, Wnd wnd)
    {
        super(source);
        _code = code;
        _message = message;
        _paramL = paramL;
        _paramH = paramH;
        _time = time;
        _wnd = wnd;
    }

    /**
     * Returns value that specifies how to process the message.
     *
     * @return Value that specifies how to process the message.
     */
    public long getCode()
    {
        return _code;
    }

    /**
     * Returns value that indicates that the hook procedure must record the contents
     *  of this instance by copying them to a buffer or file.
     * 
     * @return true if the hook procedure must record the contents
     *  of this instance by copying them to a buffer or file. Otherwise false.
     */
    public boolean isAction()
    {
        return _code == 0;
    }

    /**
     * Returns value that indicates that a system-modal dialog box has been destroyed.
     * The hook procedure must resume recording.
     * 
     * @return true if a system-modal dialog box has been destroyed. Otherwise false.
     */
    public boolean isSystemModalDialogDestroyed()
    {
        return _code == HC_SYSMODALOFF;
    }

    /**
     * Returns value that indicates that a system-modal dialog box is being displayed.
     * Until the dialog box is destroyed, the hook procedure must stop recording.
     *
     * @return true if the system-modal dialog box is being displayed. Otherwise false.
     */
    public boolean isSystemModalDialogDisplayed()
    {
        return _code == HC_SYSMODALON;
    }

    /**
     * Returns the message.
     *
     * @return The message.
     */
    public long getMessage()
    {
        return _message;
    }

    /**
     * Returns value that specifies additional information about the message.
     *
     * @return Value that specifies additional information about the message.
     */
    public long getParamL()
    {
        return _paramL;
    }

    /**
     * Returns value that specifies additional information about the message.
     *
     * @return Value that specifies additional information about the message.
     */
    public long getParamH()
    {
        return _paramH;
    }

    /**
     * Returns the time at which the message was posted.
     *
     * @return The time at which the message was posted.
     */
    public long getTime()
    {
        return _time;
    }

    /**
     * Returns handle to the window to which the message was posted.
     *
     * @return Handle to the window to which the message was posted.
     */
    public Wnd getWnd()
    {
        return _wnd;
    }
}