/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.shell;

import com.jniwrapper.*;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.util.Enums;

import java.net.MalformedURLException;
import java.net.URL;

/**
 * Represents <code>COMPONENT</code> native structure.
 * <a href = "http://msdn.microsoft.com/en-us/library/bb773225(VS.85).aspx"> MSDN Documentation </a>
 */
public class ActiveDesktopItem extends Structure
{
    private static final int MAX_PATH = 260;
    private static final int INTERNET_MAX_SCHEME_LENGTH = 32;
    private static final int INTERNET_MAX_PATH_LENGTH = 2048;

    private static final int INTERNET_MAX_URL_LENGTH = INTERNET_MAX_SCHEME_LENGTH + "://\0".length() + INTERNET_MAX_PATH_LENGTH;

    private UInt32 dwSize = new UInt32();
    private UInt32 dwID = new UInt32();
    private Int iComponentType = new Int();
    private IntBool fChecked = new IntBool();
    private IntBool fDirty = new IntBool();
    private IntBool fNoScroll = new IntBool();
    private ComponentLocation cpPos = new ComponentLocation();
    private WideString wszFriendlyName = new WideString(MAX_PATH);
    private WideString wszSource = new WideString(INTERNET_MAX_URL_LENGTH);
    private WideString wszSubscribedURL = new WideString(INTERNET_MAX_URL_LENGTH);

    /**
     *  The class, which represents the type of the component
     */
    public static class ComponentType extends EnumItem
    {
        /**
         * Component is HTML document
         */
        public static ComponentType HTMLDOC = new ComponentType(0);

        /**
         * Component is picture
         */
        public static ComponentType PICTURE = new ComponentType(1);

        /**
         * Component is Web site
         */
        public static ComponentType WEBSITE = new ComponentType(2);

        /**
         * Component is Microsoft ActiveX control.
         */
        public static ComponentType CONTROL = new ComponentType(3);


        /**
         * Component is HTML document in Compressed format
         */
        public static ComponentType CFHTML = new ComponentType(4);

        /**
         * Creates an instance of ComponentType, wHich represents the type of the component
         *
         * @param value Component type
         */
        private ComponentType(int value)
        {
            super(value);

        }
    }

    /**
     * Creates an instance of ActiveDesktopItem class
     */
    public ActiveDesktopItem()
    {
        init(new Parameter[]{dwSize, dwID, iComponentType, fChecked, fDirty, fNoScroll, cpPos, wszFriendlyName, wszSource, wszSubscribedURL});
        dwSize.setValue(getLength());
        dwID.setValue(0);
    }

    /**
     * Creates an instance of ActiveDesktopItem with a specified item.
     *
     * @param that Instance of the ActiveDesktopItem class
     */
    public ActiveDesktopItem(ActiveDesktopItem that)
    {
        this();
        initFrom(that);
    }

    /**
     * Returns the component type
     *
     * @return The component type
     */
    public ComponentType getComponentType()
    {
        return (ComponentType)Enums.getItem(ComponentType.class, (int)iComponentType.getValue());
    }

    /**
     *  Specifies the component type
     *
     * @param value the component type
    */
    public void setComponentType(ComponentType value)
    {
        iComponentType.setValue(value.getValue());
    }

    /**
     * Returns the value that indicates whether the component is enabled or not
     *
     * @return Value that is set to TRUE if the component is enabled, or FALSE if it's not
     */
    public boolean getChecked()
    {
        return fChecked.getBooleanValue();
    }

    /**
     * Switches component to enable or disable state
     * 
     * @param checked Value that is set to TRUE if the component is enabled, or FALSE if it's not
     */
    public void setChecked(boolean checked)
    {
        fChecked.setBooleanValue(checked);
    }

    /**
     *  Returns value that is set to TRUE if the component has been modified and not yet saved to disk.
     *  It will be set to FALSE if the component has not been modified, or if it has been modified and saved to disk.
     *
     * @return Value that is set to TRUE if the component has been modified and not yet saved to disk. It will be set to FALSE if the component has not been modified, or if it has been modified and saved to disk
     */
    public boolean getDirty()
    {
        return fDirty.getBooleanValue();
    }


    /**
     * Sets the value of the property <code>dirty</code>
     *
     * @param dirty Value that is set to TRUE if the component has been modified and not yet saved to disk. It will be set to FALSE if the component has not been modified, or if it has been modified and saved to disk
     */
    public void setDirty(boolean dirty)
    {
        fDirty.setBooleanValue(dirty);
    }

    /**
     * Returns value that is set to TRUE if the component is scrollable, or FALSE if not
     *
     * @return Value that is set to TRUE if the component is scrollable, or FALSE if not
     */
    public boolean getScroll()
    {
        return fNoScroll.getBooleanValue();
    }

    /**
     * Sets the value of the property, which indicates whether the component is scrollable or not
     *
     * @param scroll Value that is set to TRUE if the component is scrollable, or FALSE if not
     */
    public void setScroll(boolean scroll)
    {
        fNoScroll.setBooleanValue(scroll);
    }

    /**
     * Returns a {@link com.jniwrapper.win32.shell.ComponentLocation} structure that contains the position and size information.
     *
     * @return {@link com.jniwrapper.win32.shell.ComponentLocation} structure that contains the position and size information.
     */
    public ComponentLocation getLocation()
    {
        return cpPos;
    }

    /**
     * Sets the value of the property that contains the position and size information.
     * @param location {@link com.jniwrapper.win32.shell.ComponentLocation} structure that contains the position and size information.
     */
    public void setLocation(ComponentLocation location)
    {
        cpPos.copyFrom(location);
    }

    /**
     * Returns the friendly name of the component
     *
     * @return Friendly name of the component
     */
    public String getFriendlyName()
    {
        return wszFriendlyName.getValue();
    }

    /**
     * Sets friendly name of the component
     * 
     * @param friendlyName Friendly name of the component
     */
    public void setFriendlyName(String friendlyName)
    {
        wszFriendlyName.setValue(friendlyName);
    }

    /**
     * Returns component's URL
     *
     * @return component's URL
     */
    public URL getSource() throws MalformedURLException
    {
        return new URL(wszSource.getValue());
    }

    /**
     * Sets the URL of the component
     *
     * @param source Component's URL
     */
    public void setSource(URL source)
    {
        wszSource.setValue(source.toString());
    }

    /**
     * Returns subscribed component's URL
     *
     * @return subscribed component's URL
     */
    public URL getSubscribedURL() throws MalformedURLException
    {
        return new URL(wszSubscribedURL.getValue());
    }

    /**
     * Sets subscribed URL of the component
     *
     * @param subscribedURL Subscribed component's URL
     */
    public void setSubscribedURL(URL subscribedURL)
    {
        wszSubscribedURL.setValue(subscribedURL.toString());
    }

    /**
     * Creates copy of this instance
     *
     * @return Copy of this instance
     */
    public Object clone()
    {
        return new ActiveDesktopItem(this);
    }
}