/*
 * Copyright (c) 2000-2009 TeamDev Ltd. All rights reserved.
 * TeamDev PROPRIETARY and CONFIDENTIAL.
 * Use is subject to license terms.
 */
package com.jniwrapper.win32.ui;

import com.jniwrapper.Parameter;
import com.jniwrapper.Structure;
import com.jniwrapper.UInt;
import com.jniwrapper.UInt32;
import com.jniwrapper.util.FlagSet;

/**
 * This class represents <code>FLASHWINFO</code> native structure.
 * 
 * @author Serge Piletsky
 */
public class FlashInfo extends Structure
{
    private UInt _size = new UInt();
    private Wnd _wnd;
    private UInt32 _flags = new UInt32();
    private UInt _count = new UInt();
    private UInt32 _timeout = new UInt32();

    /**
     * Creates an instance of flash information used in the {@link Wnd} class for the
     * <code>flashWindow</code> function.
     * 
     * @param wnd a window to be flashed.
     * @param options the flash status for a window.
     * @param count the number of times to flash the window.
     * @param timeout a flash rate in milliseconds; 0 is the default system flash
     * rate.
     */
    public FlashInfo(Wnd wnd, FlashOptions options, long count, long timeout)
    {
        _wnd = wnd;
        _flags.setValue(options.getFlags());
        _count.setValue(count);
        _timeout.setValue(timeout);
        init(new Parameter[] {_size, _wnd, _flags, _count, _timeout}, (short)8);
        _size.setValue(getLength());
    }

    /**
     * Creates instance of the <code>FlashInfo</code>
     * @param that
     */
    public FlashInfo(FlashInfo that)
    {
        this(that._wnd, new FlashOptions(that._flags.getValue()), that._count.getValue(), that._timeout.getValue());
        initFrom(that);
    }

    /**
     * Returns the copy of this object
     * @return The copy of this object 
     */
    public Object clone()
    {
        return new FlashInfo(this);
    }

    public static class FlashOptions extends FlagSet
    {
        /**
         * Stop flashing. The system restores the window to its original state.
         */
        public static final int STOP         = 0;
        /**
         * Flash the window caption.
         */
        public static final int CAPTION      = 1;
        /**
         * Flash the taskbar button.
         */
        public static final int TRAY         = 2;
        /**
         * Flash both the window caption and taskbar button. This is equivalent to setting the
         * {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#CAPTION}  FLASHW_CAPTION and
         * {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TRAY} flags.
         */
        public static final int ALL          = CAPTION | TRAY;
        /**
         * Flash continuously, until the FLASHW_STOP flag is set.
         */
        public static final int TIMER        = 4;
        /**
         * Flash continuously until the window comes to the foreground.
         */
        public static final int TIMERNOFG    = 0xC;

        /**
         * Creates the instance of the <code>FlashOptions</code>
         */
        public FlashOptions()
        {
        }

        /**
         * Creates the instance of the <code>FlashOptions</code> with the specified flags
         *
         * @param flags Flags
         */
        public FlashOptions(long flags)
        {
            super(flags);
        }

        /**
         * Sets options to stop flashing the window.
         */
        public void setStop()
        {
            clear();
        }

        /**
         * Sets options to flash the window caption.
         * 
         * @param value if true, only the caption flashes; if false,
         * the caption flashing is off.
         */
        public void setFlashCaption(boolean value)
        {
            if (value)
            {
                add(CAPTION);
            }
            else
            {
                remove(CAPTION);
            }
        }

        /**
         * Returns value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#CAPTION} flag
         *
         * @return Value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#CAPTION} flag
         */
        public boolean isFlashCaption()
        {
            return contains(CAPTION);
        }

        /**
         * Sets options to flash the taskbar button.
         * 
         * @param value if true, the taskbar button flashes in the tray; if false, 
         * the taskbar button flashing is off.
         */
        public void setFlashTray(boolean value)
        {
            if (value)
            {
                add(TRAY);
            }
            else
            {
                remove(TRAY);
            }
        }

        /**
         * Returns value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TRAY} flag
         *
         * @return Value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TRAY} flag
         */
        public boolean isFlashTray()
        {
            return contains(TRAY);
        }

        /**
         * Sets options to flash both the window caption and taskbar button.
         * 
         * @param value if true, to flash the caption and taskbar button; if
         * false flashing of the caption and taskbar button is off.
         */
        public void setFlashAll(boolean value)
        {
            if (value)
            {
                add(ALL);
            }
            else
            {
                remove(ALL);
            }
        }

        /**
         * Returns value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#ALL} flag
         *
         * @return Value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#ALL} flag
         */
        public boolean isFlashAll()
        {
            return contains(ALL);
        }

        /**
         * Sets options to flash the window continuously until the
         * <code>setStop()</code> flag is set.
         * 
         * @param value if true, to flash the window continuously.
         */
        public void setFlashContinuously(boolean value)
        {
            if (value)
            {
                add(TIMER);
            }
            else
            {
                remove(TIMER);
            }
        }

        /**
         * Returns value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TIMER} flag
         *
         * @return Value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TIMER} flag
         */
        public boolean isFlashContinuously()
        {
            return contains(TIMER);
        }

        /**
         * Sets options to flash the window continuously until the window comes
         * to the foreground.
         * 
         * @param value if true, to flash the window continuously.
         */
        public void setFlashContinuouslyNoForeground(boolean value)
        {
            if (value)
            {
                add(TIMERNOFG);
            }
            else
            {
                remove(TIMERNOFG);
            }
        }

        /**
         * Returns value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TIMERNOFG} flag
         *
         * @return Value of the {@link com.jniwrapper.win32.ui.FlashInfo.FlashOptions#TIMERNOFG} flag
         */
        public boolean isFlashContinuouslyNoForeground()
        {
            return contains(TIMERNOFG);
        }
    }
}