// The xMule Project - A Peer-2-Peer File Sharing Program
//
// Copyright (C) 2003-2006 Theodore R. Smith ( hopeseekr@gmail.com / http://www.xmule.ws/ )
// Copyright (C) 2002 Merkur ( devs@emule-project.net / http://www.emule-project.net )
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of Version 2 of the GNU General Public
// License as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif // HAVE_CONFIG_H

// Test if we have _GNU_SOURCE before the next step will mess up
// setting __USE_GNU
// (only needed for gcc-2.95 compatibility, gcc 3.2 always defines it)
#include "wx/setup.h"
#include <errno.h>
#include <string.h>
#include <iostream>                         // std::cout

using std::cout;
using std::endl;

// Mario Sergio Fujikawa Ferreira <lioux@FreeBSD.org>
// to detect if this is a *BSD system
#if defined(HAVE_SYS_PARAM_H)
#include <sys/param.h>
#endif

// test message
#ifdef __GNUG__
#	pragma implementation "CFile.h"
#endif

// For compilers that support precompilation, includes "wx.h".
#include "wx/wxprec.h"
#include "CFile.h"
#include "xmule.h"
#include "xmuleDlg.h"

#ifdef __BORLANDC__
#	pragma hdrstop
#endif

// standard
#if defined(__WXMSW__) && !defined(__GNUWIN32__) && !defined(__WXWINE__) && !defined(__WXMICROWIN__)
#	include  <io.h>

#ifndef __SALFORDC__
#	define   WIN32_LEAN_AND_MEAN
#	define   NOSERVICE
#	define   NOIME
#	define   NOATOM
#	define   NOGDI
#	define   NOGDICAPMASKS
#	define   NOMETAFILE
#	define   NOMINMAX
#	define   NOMSG
#	define   NOOPENFILE
#	define   NORASTEROPS
#	define   NOSCROLL
#	define   NOSOUND
#	define   NOSYSMETRICS
#	define   NOTEXTMETRIC
#	define   NOWH
#	define   NOCOMM
#	define   NOKANJI
#	define   NOCRYPT
#	define   NOMCX
#endif

#elif (defined(__UNIX__) || defined(__GNUWIN32__))
#	include  <unistd.h>
#	ifdef __GNUWIN32__
#		include <windows.h>
#	endif
#elif defined(__DOS__)
#	if defined(__WATCOMC__)
#		include <io.h>
#	elif defined(__DJGPP__)
#		include <io.h>
#		include <unistd.h>
#		include <stdio.h>
#	else
#		error  "Please specify the header with file functions declarations."
#	endif
#elif (defined(__WXPM__))
#	include <io.h>
#elif (defined(__WXSTUBS__))
// Have to ifdef this for different environments
#	include <io.h>
#elif (defined(__WXMAC__))
#if __MSL__ < 0x6000
int access(const char * path, int mode)
{
    return 0;
}

#else
int _access(const char * path, int mode)
{
    return 0;
}

#endif
char * mktemp(char * path)
{
    return path;
}

#	include <stat.h>
#	include  <unistd.h>
#else
#	error  "Please specify the header with file functions declarations."
#endif  //Win/UNIX

#include  <stdio.h>       // SEEK_xxx constants
#include  <fcntl.h>       // O_RDONLY &c

#if !defined(__MWERKS__) || defined(__WXMSW__)
#	include  <sys/types.h>   // needed for stat
#	include  <sys/stat.h>    // stat
#endif

// Windows compilers don't have these constants
#ifndef W_OK
enum
{
    // test for existence:
    F_OK = 0,
    //          execute permission:
    X_OK = 1,
    //          write:
    W_OK = 2,
    //          read:
    R_OK = 4
}

;

#endif // W_OK

// there is no distinction between text and binary files under Unix, so define
// O_BINARY as 0 if the system headers don't do it already
#if defined(__UNIX__) && !defined(O_BINARY)
#	define   O_BINARY    (0)
#endif  //__UNIX__

#ifdef __SALFORDC__
#	include <unix.h>
#endif

#ifndef MAX_PATH
// *BSD compatibility
#	if (defined(BSD) && (BSD >= 199103))
#		define MAX_PATH MAXPATHLEN
#	else
#		define MAX_PATH 512
#	endif
#endif

// some broken compilers don't have 3rd argument in open() and creat()
#ifdef __SALFORDC__
#	define ACCESS(access)
#	define stat    _stat
#else // normal compiler
#	define ACCESS(access)  , (access)
#endif // Salford C

// wxWindows
#ifndef WX_PRECOMP
#	include  "wx/string.h"
#	include  "wx/intl.h"
#	include  "wx/log.h"
#endif // !WX_PRECOMP

#include  "wx/filename.h"

#include  "wx/filefn.h"

#ifdef __WXMSW__
#	include "wx/msw/mslu.h"
#endif

// ============================================================================
// implementation of CFile
// ============================================================================

// ----------------------------------------------------------------------------
// static functions
// ----------------------------------------------------------------------------

bool CFile::Exists(const wxChar * name)
{
    return wxFileExists(name);
}

bool CFile::Access(const wxChar * name, OpenMode mode)
{
    int how;
    switch (mode)
    {
    default:
        wxFAIL_MSG(wxT("bad CFile::Access mode parameter."));
        // fall through
    case read:
        how = R_OK;
        break;
    case write:
        how = W_OK;
        break;
    case read_write:
        how = R_OK | W_OK;
        break;
    }
    return wxAccess(name, how) == 0;
}

// ----------------------------------------------------------------------------
// opening/closing
// ----------------------------------------------------------------------------

// ctors
CFile::CFile(const wxChar * szFileName, OpenMode mode)
{
    m_fd = fd_invalid;
    m_error = FALSE;
    Open(szFileName, mode);
}

// create the file, fail if it already exists and bOverwrite
bool CFile::Create(const wxChar * szFileName, bool bOverwrite, int accessMode)
{
    fFilePath = szFileName;
    // if bOverwrite we create a new file or truncate the existing one,
    // otherwise we only create the new file and fail if it already exists
#if defined(__WXMAC__) && !defined(__UNIX__)
    // Dominic Mazzoni [dmazzoni+@cs.cmu.edu] reports that open is still broken on the mac, so we replace
    // int fd = open(wxUnix2MacFilename( szFileName ), O_CREAT | (bOverwrite ? O_TRUNC : O_EXCL), access);
    int fd = creat(szFileName, accessMode);
#else
    int fd = wxOpen(szFileName,
    O_BINARY | O_WRONLY | O_CREAT |
    (bOverwrite ? O_TRUNC: O_EXCL)
    ACCESS(accessMode));
#endif
    if (fd == - 1)
    {
        wxLogSysError(_("can't create file '%s'"), szFileName);
        return FALSE;
    }
    else
    {
        Attach(fd);
        return TRUE;
    }
}

// open the file
bool CFile::Open(const wxChar * szFileName, OpenMode mode, int accessMode)
{
    int flags = O_BINARY;
    fFilePath = szFileName;
    switch (mode)
    {
    case read:
        flags |= O_RDONLY;
        break;
    case write_append:
        if (CFile::Exists(szFileName))
        {
            flags |= O_WRONLY | O_APPEND;
            break;
        }
        //else: fall through as write_append is the same as write if the
        //      file doesn't exist
    case write:
        flags |= O_WRONLY | O_CREAT | O_TRUNC;
        break;
    case write_excl:
        flags |= O_WRONLY | O_CREAT | O_EXCL;
        break;
    case read_write:
        flags |= O_RDWR;
        break;
    }
    int fd = wxOpen(szFileName, flags ACCESS(accessMode));
    if (fd == - 1)
    {
        theApp.xmuledlg->AddLogLine(true, _("Can't open file '%s'"), szFileName);
        return FALSE;
    }
    else
    {
        Attach(fd);
        return TRUE;
    }
}

// close
bool CFile::Close()
{
    if (IsOpened())
    {
        if (close(m_fd) == - 1)
        {
            wxLogSysError(_("can't close file descriptor %d"), m_fd);
            m_fd = fd_invalid;
            return FALSE;
        }
        else
        m_fd = fd_invalid;
    }
    return TRUE;
}

// ----------------------------------------------------------------------------
// read/write
// ----------------------------------------------------------------------------

off_t CFile::Readpos(void * pBuf, long pos, off_t nCount)
{
    int iRc = lseek(m_fd, pos, SEEK_SET);
    if (iRc == pos)
    {
        return Read(pBuf, nCount);
    }
    else
    {
        return 0;
    }
}

// read
off_t CFile::Read(void * pBuf, off_t nCount)
{
    wxCHECK((pBuf != NULL) && IsOpened(), 0);
#ifdef __MWERKS__
    int iRc =::read(m_fd, (char *) pBuf, nCount);
#else
    int iRc =::read(m_fd, pBuf, nCount);
#endif
    if (iRc == - 1)
    {
        wxLogSysError(_("can't read from file descriptor %d"), m_fd);
        return wxInvalidOffset;
    }
    else
    return(size_t) iRc;
}

// write
size_t CFile::Writepos(const void * pBuf, long pos, size_t nCount)
{
    int iRc = lseek(m_fd, pos, SEEK_SET);
    if (iRc == pos)
    {
        return Write(pBuf, nCount);
    }
    else
    {
        return 0;
    }
}

size_t CFile::Write(const void * pBuf, size_t nCount)
{
    wxCHECK((pBuf != NULL) && IsOpened(), 0);
#ifdef __MWERKS__
#if __MSL__ >= 0x6000
    int iRc =::write(m_fd, (void *) pBuf, nCount);
#else
    int iRc =::write(m_fd, (const char *) pBuf, nCount);
#endif
#else
int iRc =::write(m_fd, pBuf, nCount);

#endif
if (iRc == - 1)
{
    wxLogSysError(_("can't write to file descriptor %d"), m_fd);
    m_error = TRUE;
    return 0;
}

else
return iRc;

}

// flush
bool CFile::Flush()
{
    if (IsOpened())
    {
        if (fsync(m_fd) == - 1)
        {
            wxLogSysError(_("can't flush file descriptor %d"), m_fd);
            return FALSE;
        }
    }
    return TRUE;
}

// ----------------------------------------------------------------------------
// seek
// ----------------------------------------------------------------------------

// seek
off_t CFile::Seek(off_t ofs, wxSeekMode mode)
{
    wxASSERT(IsOpened());
    int origin;
    switch (mode)
    {
    default:
        wxFAIL_MSG(_("unknown seek origin"));
    case wxFromStart:
        origin = SEEK_SET;
        break;
    case wxFromCurrent:
        origin = SEEK_CUR;
        break;
    case wxFromEnd:
        origin = SEEK_END;
        break;
    }
    int iRc = lseek(m_fd, ofs, origin);
    if (iRc == - 1)
    {
        cout << "Error in lseek: " << strerror(errno) << endl;
        return wxInvalidOffset;
    }
    else
    return(off_t) iRc;
}

// get current off_t
off_t CFile::Tell() const
{
    wxASSERT(IsOpened());
    int iRc = wxTell(m_fd);
    if (iRc == - 1)
    {
        wxLogSysError(_("can't get seek position on file descriptor %d"), m_fd);
        return wxInvalidOffset;
    }
    else
    return(off_t) iRc;
}

// get current file length
off_t CFile::Length() const
{
    wxASSERT(IsOpened());
#ifdef __VISUALC__
    int iRc = _filelength(m_fd);
#else // !VC++
    int iRc = wxTell(m_fd);
    if (iRc != - 1)
    {
        // @ have to use const_cast :-(
        int iLen = ((CFile *) this)->SeekEnd();
        if (iLen != - 1)
        {
            // restore old position
            if (((CFile *) this)->Seek(iRc) == - 1)
            {
                // error
                iLen = - 1;
            }
        }
        iRc = iLen;
    }
#endif  // VC++
    if (iRc == - 1)
    {
        wxLogSysError(_("can't find length of file on file descriptor %d"), m_fd);
        return wxInvalidOffset;
    }
    else
    return(off_t) iRc;
}

// is end of file reached?
bool CFile::Eof() const
{
    wxASSERT(IsOpened());
    int iRc;
#if defined(__DOS__) || defined(__UNIX__) || defined(__GNUWIN32__) || defined( __MWERKS__ ) || defined(__SALFORDC__)
    // @@ this doesn't work, of course, on unseekable file descriptors
    off_t ofsCur = Tell(),
    ofsMax = Length();
    if (ofsCur == wxInvalidOffset || ofsMax == wxInvalidOffset)
    iRc = - 1;
    else
    iRc = ofsCur == ofsMax;
#else  // Windows and "native" compiler
    iRc = eof(m_fd);
#endif // Windows/Unix
    switch (iRc)
    {
    case 1:
        break;
    case 0:
        return FALSE;
    case - 1:
        wxLogSysError(_("can't determine if the end of file is reached on descriptor %d"), m_fd);
        break;
    default:
        wxFAIL_MSG(_("invalid eof() return value."));
    }
    return TRUE;
}

