// The xMule Project - A Peer-2-Peer File Sharing Program
//
// Copyright (C) 2003-2006 Theodore R. Smith ( hopeseekr@gmail.com / http://www.xmule.ws/ )
// Copyright (C) 2002 Merkur ( devs@emule-project.net / http://www.emule-project.net )
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of Version 2 of the GNU General Public
// License as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

#ifdef PRECOMP
#	include "xmule-headers.h"
#else
#	include "ED2KLink.h"
#	include "otherfunctions.h"
#	include "resource.h"
#	include "wintypes.h"
#	include "xmule.h"
#endif

#include <wx/datetime.h>
#include <sstream>

using std::endl;

#if 1
namespace
{
    struct autoFree
    {
        autoFree(TCHAR *p): m_p(p)
        {
        }
        ~autoFree()
        {
            free(m_p);
        }
        private:
        TCHAR *m_p;
        };
    inline unsigned int FromHexDigit(TCHAR digit)
    {
        switch (digit)
        {
        case('0'): return 0;
        case('1'): return 1;
        case('2'): return 2;
        case('3'): return 3;
        case('4'): return 4;
        case('5'): return 5;
        case('6'): return 6;
        case('7'): return 7;
        case('8'): return 8;
        case('9'): return 9;
        case('A'): return 10;
        case('B'): return 11;
        case('C'): return 12;
        case('D'): return 13;
        case('E'): return 14;
        case('F'): return 15;
        case('a'): return 10;
        case('b'): return 11;
        case('c'): return 12;
        case('d'): return 13;
        case('e'): return 14;
        case('f'): return 15;
        default: throw GetResString(IDS_ERR_ILLFORMEDHASH);
        }
    }
}

#endif

CED2KLink::~CED2KLink()
{
}

/////////////////////////////////////////////
// CED2KServerListLink implementation
/////////////////////////////////////////////
CED2KServerListLink::CED2KServerListLink(const TCHAR *address)
{
    m_address = wxString(address, *wxConvCurrent);
}

CED2KServerListLink::~CED2KServerListLink()
{
}

void
CED2KServerListLink::GetLink(wxString &lnk)
{
    lnk = wxT("ed2k://|serverlist|");
    lnk += m_address;
    lnk += wxT("|/");
}

CED2KServerListLink *
CED2KServerListLink::GetServerListLink()
{
    return this;
}

CED2KServerLink *
CED2KServerListLink::GetServerLink()
{
    return 0;
}

CED2KFileLink *
CED2KServerListLink::GetFileLink()
{
    return 0;
}

CED2KLink::LinkType
CED2KServerListLink::GetKind() const
{
    return kServerList;
}

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/////////////////////////////////////////////
// CED2KServerLink implementation
/////////////////////////////////////////////
CED2KServerLink::CED2KServerLink(const TCHAR *ip, const TCHAR *port)
{
    m_ip = inet_addr(ip);
    std::ostringstream internalbuffer;
    //_tcstoul(port,0,10);:
    unsigned long ul = atoi(port);
    if (ul > 0xFFFF)
    throw GetResString(IDS_ERR_BADPORT);
    m_port = static_cast < uint16_t > (ul);
    internalbuffer << "Server " << ip << ":" << port;
    m_defaultName = wxString(internalbuffer.str().c_str(), *wxConvCurrent);
}

CED2KServerLink::~CED2KServerLink()
{
}

void
CED2KServerLink::GetLink(wxString &lnk)
{
    in_addr adr;
    std::ostringstream internalbuffer;
    adr.s_addr = m_ip;
    internalbuffer << "ed2k://|server|" << inet_ntoa(adr) << "|" << static_cast < int > (m_port) << "|/";
    lnk = wxString(internalbuffer.str().c_str(), *wxConvCurrent);
}

CED2KServerListLink *
CED2KServerLink::GetServerListLink()
{
    return 0;
}

CED2KServerLink *
CED2KServerLink::GetServerLink()
{
    return this;
}

CED2KFileLink *
CED2KServerLink::GetFileLink()
{
    return 0;
}

CED2KLink::LinkType
CED2KServerLink::GetKind() const
{
    return kServer;
}

/////////////////////////////////////////////
// CED2KFileLink implementation
/////////////////////////////////////////////
CED2KFileLink::CED2KFileLink(const TCHAR *name, const TCHAR *size, const TCHAR *hash, const TCHAR *sources)
: m_name(wxString(name, *wxConvCurrent))
, m_size(wxString(size, *wxConvCurrent))
{
    SourcesList = NULL;
    if (strlen(hash) != 32)
    throw GetResString(IDS_ERR_ILLFORMEDHASH);
    for (int idx = 0 ; idx < 16 ;++idx)
    {
        m_hash[idx] = FromHexDigit( *hash++) *16;
        m_hash[idx] += FromHexDigit( *hash++);
    }
    if (sources)
    {
        TCHAR *pNewString = strdup(sources);
        autoFree liberator(pNewString);
        TCHAR *pCh = pNewString;
        TCHAR *pEnd;
        TCHAR *pIP;
        TCHAR *pPort;
        bool bAllowSources;
        TCHAR date[3];
        //COleDateTime expirationDate;
        wxDateTime expirationDate;
        int nYear, nMonth, nDay;
        uint16_t nCount = 0;
        uint32_t dwID;
        uint16_t nPort;
        uint32_t dwServerIP = 0;
        uint16_t nServerPort = 0;
        unsigned long ul;
        int nInvalid = 0;
        pCh = strstr(pCh, "sources");
        if (pCh != NULL)
        {
            // point to char after "sources":
            pCh = pCh + 7;
            pEnd = pCh;
            // make pEnd point to the terminating NULL:
            while ( *pEnd) pEnd++;
            bAllowSources = true;
            // if there's an expiration date...
            if ( *pCh == '@' && (pEnd - pCh) > 7)
            {
                // after '@':
                pCh++;
                // terminate the two character string:
                date[2] = 0;
                date[0] = * (pCh++);
                date[1] = * (pCh++);
                nYear = atol(date) + 2000;
                date[0] = * (pCh++);
                date[1] = * (pCh++);
                nMonth = atol(date);
                date[0] = * (pCh++);
                date[1] = * (pCh++);
                nDay = atol(date);
                expirationDate.Set(nYear, (wxDateTime::Month) nMonth, nDay, 0, 0, 0, 0);
                bAllowSources = expirationDate.IsValid();
                if (bAllowSources) bAllowSources = (wxDateTime::UNow() < expirationDate);
            }
            // increment pCh to point to the first "ip:port" and check for sources
            if (bAllowSources &&++pCh < pEnd)
            {
                SourcesList = new CMemFile();
                // init to 0, we'll fix this at the end.:
                SourcesList->Write( &nCount, sizeof(nCount));
                // for each "ip:port" source string until the end
                // limit to prevent overflow (uint16_t due to CPartFile::AddClientSources)
#define MAXSHORT 65535
                while ( *pCh != 0 &&nCount < MAXSHORT)
                {
                    pIP = pCh;
                    // find the end of this ip:port string & start of next ip:port string.
                    if ((pCh = strchr(pCh, ',')))
                    {
                        // terminate current "ip:port":
                        *pCh = 0;
                        // point to next "ip:port":
                        pCh++;
                    }
                    else
                    pCh = pEnd;
                    // if port is not present for this ip, go to the next ip.
                    if ((pPort = strchr(pIP, _T(':'))) == NULL)
                    {
                        nInvalid++;
                        continue;
                    }
                    // terminate ip string:
                    *pPort = 0;
                    // point pPort to port string.:
                    pPort++;
                    dwID = inet_addr(pIP);
                    //tcstoul( pPort, 0, 10 );:
                    ul = atoi(pPort);
                    nPort = static_cast < uint16_t > (ul);
                    // skip bad ips / ports
                    if (dwID == INADDR_NONE || dwID < 16777216 ||
                    ul > 0xFFFF || ul == 0)
                    {
                        nInvalid++;
                        continue;
                    }
                    SourcesList->Write( &dwID, sizeof(dwID));
                    SourcesList->Write( &nPort, sizeof(nPort));
                    SourcesList->Write( &dwServerIP, sizeof(dwServerIP));
                    SourcesList->Write( &nServerPort, sizeof(nServerPort));
                    nCount++;
                }
                //ToBegin();:
                SourcesList->Seek(0);
                SourcesList->Write( &nCount, sizeof(nCount));
                //ToBegin();:
                SourcesList->Seek(0);
                if (nCount == 0)
                {
                    delete SourcesList;
                    SourcesList = NULL;
                }
            }
        }
    }
}

CED2KFileLink::~CED2KFileLink()
{
}

void
CED2KFileLink::GetLink(wxString &lnk)
{
    std::ostringstream internalbuffer;
    internalbuffer << "ed2k://|file|" << m_name.mb_str(*wxConvCurrent) << "|" << m_size.mb_str(*wxConvCurrent) << "|";
    for (int idx = 0 ; idx != 16 ;++idx)
    {
        unsigned int ui1 = m_hash[idx] / 16;
        unsigned int ui2 = m_hash[idx] % 16;
        internalbuffer << static_cast< TCHAR >(ui1 > 9 ? (('0') + ui1): (('A') + (ui1 - 10))) << static_cast< TCHAR >(ui2 > 9 ? (('0') + ui2): (('A') + (ui2 - 10)));
    }
    internalbuffer << "|/";
    lnk = wxString(internalbuffer.str().c_str(), *wxConvCurrent);
}

CED2KServerListLink *
CED2KFileLink::GetServerListLink()
{
    return 0;
}

CED2KServerLink *
CED2KFileLink::GetServerLink()
{
    return 0;
}

CED2KFileLink *
CED2KFileLink::GetFileLink()
{
    m_name.Replace(wxT("%20"), wxT(" "));
    return this;
}

CED2KLink::LinkType
CED2KFileLink::GetKind() const
{
    return kFile;
}

//static
CED2KLink *
CED2KLink::CreateLinkFromUrl(const TCHAR *uri)
{
    // Parse pseudo-URI
    const TCHAR *pChArray[7];
    if (uri == 0)
    throw wxString(_T("null ed2k link"));
    TCHAR *pNewString = strdup(uri);
    autoFree liberator(pNewString);
    TCHAR *pCh = pNewString;
    const TCHAR *pStart = pCh;
    int idx = 0;
    for (idx = 0 ; idx < 7 ; idx++) pChArray[idx] = NULL;
    idx = 0;
    while (idx < 7 && ((pCh = strchr(pCh, ('|'))) != 0))
    {
        pChArray[idx++] = pStart;
        *pCh = 0;
        ++pCh;
        pStart = pCh;
    }
    if ( *pStart != ('/'))
    {
        throw wxString((GetResString(IDS_ERR_BADED2KLINK)));
    }
    if (idx < 3
    || pChArray[0] == 0
    || pChArray[1] == 0
    || pChArray[2] == 0
    //		|| pChArray[3] == 0 // This was preventing ed2k serverlist links from working..
    || strcmp(("ed2k://"), pChArray[0]) != 0
    )
    {
        throw GetResString(IDS_ERR_BADED2KLINK);
    }
    if (strcmp(("file"), pChArray[1]) == 0 &&idx >= 5 &&pChArray[4] != 0)
    {
        return new CED2KFileLink(pChArray[2], pChArray[3], pChArray[4], pChArray[6]);
    }
    else if(strcmp(("serverlist"), pChArray[1]) == 0 &&idx == 3)
    {
        return new CED2KServerListLink(pChArray[2]);
    }
    else if(strcmp(("server"), pChArray[1]) == 0 &&idx == 4)
    {
        return new CED2KServerLink(pChArray[2], pChArray[3]);
    }
    else
    {
        throw GetResString(IDS_ERR_NOSLLINK);
    }
    return 0;
}

#if 0
//static
CED2KLink *
CED2KLink::CreateLinkFromUrl(const TCHAR *uri)
{
    // Parse pseudo-URI
    const TCHAR *pChArray[7];
    if (uri == 0)
    throw wxString(("null ed2k link"));
    TCHAR *pNewString = strdup(uri);
    autoFree liberator(pNewString);
    TCHAR *pCh = pNewString;
    const TCHAR *pStart = pCh;
    int idx = 0;
    for (idx = 0 ; idx < 7 ; idx++) pChArray[idx] = NULL;
    idx = 0;
    while (idx < 7 && ((pCh = strchr(pCh, ('|'))) != 0))
    {
        pChArray[idx++] = pStart;
        *pCh = 0;
        ++pCh;
        pStart = pCh;
    }
    if ( *pStart != ('/'))
    {
        throw wxString((GetResString(IDS_ERR_BADED2KLINK)));
    }
    if (idx < 3
    || pChArray[0] == 0
    || pChArray[1] == 0
    || pChArray[2] == 0
    || pChArray[3] == 0
    || strcmp(("ed2k://"), pChArray[0]) != 0
    )
    {
        throw GetResString(IDS_ERR_BADED2KLINK);
    }
    if (strcmp(("file"), pChArray[1]) == 0 &&idx == 5 &&pChArray[4] != 0)
    {
        return new CED2KFileLink(pChArray[2], pChArray[3], pChArray[4]);
    }
    else if(strcmp(("serverlist"), pChArray[1]) == 0 &&idx == 3)
    {
        return new CED2KServerListLink(pChArray[2]);
    }
    else if(strcmp(("server"), pChArray[1]) == 0 &&idx == 4)
    {
        return new CED2KServerLink(pChArray[2], pChArray[3]);
    }
    else
    {
        throw GetResString(IDS_ERR_NOSLLINK);
    }
    return 0;
}

#endif

