// The xMule Project - A Peer-2-Peer File Sharing Program
//
// Copyright (C) 2003-2006 Theodore R. Smith ( hopeseekr@gmail.com / http://www.xmule.ws/ )
// Copyright (C) 2002 Merkur ( devs@emule-project.net / http://www.emule-project.net )
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of Version 2 of the GNU General Public
// License as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

#include "config.h"

// Test if we have _GNU_SOURCE before the next step will mess up
// setting __USE_GNU
// (only needed for gcc-2.95 compatibility, gcc 3.2 always defines it)
#include "wx/setup.h"

// Mario Sergio Fujikawa Ferreira <lioux@FreeBSD.org>
// to detect if this is a *BSD system
#if defined(HAVE_SYS_PARAM_H)
#include <sys/param.h>
#endif

#ifdef PRECOMP
#	include "xmule-headers.h"
#else
#	include "Preferences.h"
#	include "opcodes.h"
#	include "otherfunctions.h"
#	include "stdlib.h"
#	include "stdio.h"
#	include "resource.h"
#include "xmule.h"                          // Needed for theApp - audited 4 Nov 2004
#endif

#include <DynPrefs/DynPrefs.h>              // Needed for DynamicPreferences

extern "C"
{
    #include <ini/iniparser.h>                  // Needed for iniparser_load
}

#include <cstdlib>
#include <ctime>
#include <fstream>                          // Needed for std::ofstream std::endl
#include <iomanip>                          // std::hex std::setprecision
#include <iostream>
#include <locale.h>
#include <sstream>                          // Needed for std::stringstream
#include <string>                           // Needed for std::string
#include <wx/filename.h>

#define ELEMENT_COUNT(X) (sizeof(X) / sizeof(X[0]))
#define RGB(a,b,c) ((a&0xff)<<16|(b&0xff)<<8|(c&0xff))

#include <sys/stat.h>
#include <sys/types.h>

#ifndef MAX_PATH
// *BSD compatibility
#if (defined(BSD) && (BSD >= 199103))
#define MAX_PATH MAXPATHLEN
#else
#define MAX_PATH 2048
#endif
#endif

#ifdef __FreeBSD__
extern long long atoll(char *s);
#endif /* __FreeBSD__ */

using std::cout;
using std::endl;

using std::hex;
using std::ifstream;
using std::ios;
using std::ofstream;
using std::stringstream;

CPreferences::CPreferences()
{
    // we need random numbers sometimes:
    srand((uint32_t) time(0));
    // Define prefs 
    prefs = new Preferences_Struct;
    memset(prefs, 0, sizeof(Preferences_Struct));
    prefsExt = new Preferences_Ext_Struct;
    memset(prefsExt, 0, sizeof(Preferences_Ext_Struct));
    // Use home directory to save preferences
    char buffer[490];
    snprintf(buffer, 400, "%s/." PACKAGE, getenv("HOME"));
    DynPrefs::Add("app-directory", wxString(buffer, *wxConvCurrent) + wxT("/"));
    mkdir(buffer, 0777);

    // Create UserHash
    CreateUserHash();
    md4cpy(&prefs->userhash, &userhash);
    if (((int *) prefs->userhash) [0] == 0 && ((int *) prefs->userhash) [1] == 0 && ((int *) prefs->userhash) [2] == 0 && ((int *) prefs->userhash) [3] == 0)
    {
       CreateUserHash();
    }
    // Load Preferences
    Load(true);
    // ???
    prefs->m_bUseSecureIdent = true;
}

CPreferences::~CPreferences()
{
    Save(5);

    Category_Struct *delcat;
    while (!catMap.IsEmpty())
    {
        delcat = catMap.GetAt(0);
        catMap.RemoveAt(0);
        delete delcat;
    }
    delete prefs;
    delete prefsExt;
}


void CPreferences::SetStandartValues()
{
    wxString buffer;
    CreateUserHash();
    md4cpy( &prefs->userhash, &userhash);
    x::WINDOWPLACEMENT defaultWPM;
    defaultWPM.length = sizeof(x::WINDOWPLACEMENT);
    defaultWPM.rcNormalPosition.left = 10;
    defaultWPM.rcNormalPosition.top = 10;
    defaultWPM.rcNormalPosition.right = 700;
    defaultWPM.rcNormalPosition.bottom = 500;
    defaultWPM.showCmd = 0;
    prefs->EmuleWindowPlacement = defaultWPM;
    prefs->versioncheckLastAutomatic = 0;
}

bool CPreferences::Load(bool startup)
{
    bool error = false;
    if(LoadPreferences())
    {
        if(startup)
	{
	    //Can't find Preferences file, but in LoadPreferences() the standard was defined. Now we have to save it.
            SetStandartValues(); 
            if(Save(1)) 
	    {
                error = true;
            }
        }
        else
	{
	    //Not at Application start. The File was deleted while the Application is running ?
            error = true;
        }
    }
    if (startup)
    {
         if(LoadCats())
         {
             //Catfile don't exist, let's create one
             if(Save(2))
	     {
	       error = true;
             }
         }
         if(Load_Shared_Dirs())
         {
	     ; //Maybe the User has not shared dirs 
         }
          if(Load_Adresses())
         {
	     ; //And also no Adresslist
         }
    }
    return error;
}

bool CPreferences::Save(unsigned char selection_mask)
{
    bool error = false;
    if (selection_mask &1)
    {
      SavePreferences();
    }
    if (selection_mask &2)
    {
       if (SaveCats())
       {
           error = true;
       }
    }
    if (selection_mask &4)
    {
       if (Save_Shared_Dirs())
       { 
           error = true;
       }
    }
    if (selection_mask &8)
    {
       if (Save_Adresses())
       {
           error = true;
       }        
    }
    return error;
}

void CPreferences::CreateUserHash()
{
    for (int i = 0 ; i != 8 ; i++)
    {
        uint16_t random = rand();
        memcpy( &userhash[i*2], &random, 2);
    }
    // mark as emule client. that will be need in later version
    userhash[5] = 14;
    userhash[14] = 111;
}

int CPreferences::GetRecommendedMaxConnections()
{
    int iRealMax =::GetMaxConnections();
    if (iRealMax == - 1 || iRealMax > 520)
    return 500;
    if (iRealMax < 20)
    return iRealMax;
    if (iRealMax <= 256)
    return iRealMax - 10;
    return iRealMax - 20;
}

x::WORD CPreferences::GetWindowsVersion()
{
    return 1;
}

uint16_t CPreferences::GetDefaultMaxConperFive()
{
    return MAXCONPER5SEC;
}


void CPreferences::ResetStatsColor(int index)
{
    switch (index)
    {
    case 0: prefs->statcolors[0] = RGB(0, 0, 64);
            break;
    case 1: prefs->statcolors[1] = RGB(192, 192, 255);
            break;
    case 2: prefs->statcolors[2] = RGB(128, 255, 128);
            break;
    case 3: prefs->statcolors[3] = RGB(0, 210, 0);
            break;
    case 4: prefs->statcolors[4] = RGB(0, 128, 0);
            break;
    case 5: prefs->statcolors[5] = RGB(255, 128, 128);
            break;
    case 6: prefs->statcolors[6] = RGB(200, 0, 0);
            break;
    case 7: prefs->statcolors[7] = RGB(140, 0, 0);
            break;
    case 8: prefs->statcolors[8] = RGB(150, 150, 255);
            break;
    case 9: prefs->statcolors[9] = RGB(192, 0, 192);
            break;
    case 10:prefs->statcolors[10] = RGB(255, 255, 128);
            break;
    case 11:prefs->statcolors[11] = RGB(0, 0, 0);
            break;
    case 12:prefs->statcolors[12] = RGB(255, 255, 255);
            break;
    default:break;
    }
}

void CPreferences::SavePreferences(void)
{
    std::string filename = getenv("HOME");
    filename += "/.xMule/prefs.ini";

    std::ofstream fout(filename.c_str());
    fout << "[xMule]" << endl;
    fout << "app-version="               << PACKAGE " " VERSION << endl;
    fout << "nickname="                  << DynPrefs::Get<wxString>("nickname").mb_str(*wxConvCurrent) << endl;
    fout << "incoming-directory="        << DynPrefs::Get<wxString>("incoming-directory").mb_str(*wxConvCurrent) << endl;
    fout << "temp-directory="            << DynPrefs::Get<wxString>("temp-directory").mb_str(*wxConvCurrent) << endl;
    fout << "MaxUpload="                << DynPrefs::Get<long>( "limits-upload") << endl;
    fout << "limits-download="           << DynPrefs::Get<long>("limits-download") << endl;
    fout << "max-connections="           << DynPrefs::Get<long>("max-connections") << endl;
    fout << "remove-dead-server="        << DynPrefs::Get<bool>("remove-dead-server") << endl;
    fout << "tcp-port="                  << DynPrefs::Get<long>("tcp-port") << endl;
    fout << "udp-port="                  << DynPrefs::Get<long>("udp-port") << endl;
    fout << "max-sources="               << DynPrefs::Get<long>("max-sources") << endl;
    fout << "Language="                 << prefs->languageID << endl;
    fout << "see-share="                 << DynPrefs::Get<long>("see-share") << endl;
    fout << "ToolTipDelay="             << prefs->m_iToolDelayTime << endl;
    fout << "graph-delay="               << DynPrefs::Get<long>("graph-delay") << endl;
    fout << "tree-delay="                << DynPrefs::Get<long>("tree-delay") << endl;
    fout << "capacities-download="       << DynPrefs::Get<long>("capacities-download") << endl;
    fout << "capacities-upload="         << DynPrefs::Get<long>("capacities-upload") << endl;
    fout << "server-connect-tries="      << DynPrefs::Get<long>("server-connect-tries") << endl;
    fout << "SplitterbarPosition="      << prefs->splitterbarPosition + 2 << endl;
    fout << "graph-average="             << DynPrefs::Get<long>("graph-average") << endl;
    fout << "max-per-5="                 << DynPrefs::Get<long>("max-per-5") << endl;
    fout << "Check4NewVersionDelay="    << prefs->versioncheckdays << endl;
    fout << "reconnect-on-loss="         << DynPrefs::Get<bool>("reconnect-on-loss") << endl;
    fout << "priority-system="           << DynPrefs::Get<bool>("priority-system") << endl;
    fout << "enable-ich="                << DynPrefs::Get<bool>("enable-ich") << endl;
    fout << "autoupdate-server="         << DynPrefs::Get<bool>("autoupdate-server") << endl;
    fout << "UpdateNotify="             << prefs->updatenotify << endl;
    fout << "minimize-to-tray="          << DynPrefs::Get<bool>("minimize-to-tray") << endl;
    fout << "add-server-server="         << DynPrefs::Get<bool>("add-server-server") << endl;
    fout << "add-server-client="         << DynPrefs::Get<bool>("add-server-client") << endl;
    fout << "show-splashscreen="         << DynPrefs::Get<bool>("show-splashscreen") << endl;
    fout << "bring-to-front="            << DynPrefs::Get<bool>("bring-to-front") << endl;
    fout << "download-doubleclick="      << DynPrefs::Get<bool>("download-doubleclick") << endl;
    fout << "beep-on-errors="            << DynPrefs::Get<bool>("beep-on-errors") << endl;
    fout << "prompt-on-exit="            << DynPrefs::Get<bool>("prompt-on-exit") << endl;
    fout << "filter-bad-ip="             << DynPrefs::Get<bool>("filter-bad-ip") << endl;
    fout << "autoconnect="               << DynPrefs::Get<bool>("autoconnect") << endl;
    fout << "online-signature="          << DynPrefs::Get<bool>("online-signature") << endl;
    fout << "start-minimized="           << DynPrefs::Get<bool>("start-minimized") << endl;
    fout << "use-safe-connect="          << DynPrefs::Get<bool>("use-safe-connect") << endl;
    fout << "xfer-on-title="             << DynPrefs::Get<bool>("xfer-on-title") << endl;
    fout << "IndicateRatings="          << prefs->indicateratings << endl;
    fout << "WatchClipboard4ED2kFilelinks=" << prefs->watchclipboard << endl;
    fout << "auto-static="               << DynPrefs::Get<bool>("auto-static") << endl;
    fout << "AutoTakeED2KLinks="        << prefs->autotakeed2klinks << endl;
    fout << "add-paused="                << DynPrefs::Get<bool>("add-paused") << endl;
    fout << "progress-bar-style="        << DynPrefs::Get<long>("progress-bar-style") << endl;
    fout << "NotifyOnDownload="         << prefs->useDownloadNotifier << endl;
    fout << "NotifyOnChat="             << prefs->useChatNotifier << endl;
    fout << "NotifyOnLog="              << prefs->useLogNotifier << endl;
    fout << "NotifierUseSound="         << prefs->useSoundInNotifier << endl;
    fout << "VideoPlayer="              << prefs->VideoPlayer << endl;
    fout << "smart-lowid-check="         << DynPrefs::Get<bool>("smart-lowid-check") << endl;
    fout << "Verbose="                  << prefs->m_bVerbose << endl;
    fout << "last-chunk-first="          << DynPrefs::Get<bool>("last-chunk-first") << endl;
    fout << "realtime-queue="            << DynPrefs::Get<bool>("realtime-queue") << endl;
    fout << "add-server-high="           << DynPrefs::Get<bool>("add-server-high") << endl;
    fout << "transfer-full-chunks="      << DynPrefs::Get<bool>("transfer-full-chunks") << endl;
    fout << "start-next-file="           << DynPrefs::Get<bool>("start-next-file") << endl;
    fout << "show-overhead="             << DynPrefs::Get<bool>("show-overhead") << endl;
    fout << "VideoPreviewBackupped="    << prefs->moviePreviewBackup << endl;
    fout << "file-buffer-size="          << DynPrefs::Get<long>("file-buffer-size") << endl;
    fout << "upload-queue-size="         << DynPrefs::Get<long>("upload-queue-size") << endl;
    fout << "auto-priority-share="       << DynPrefs::Get<bool>("auto-priority-share") << endl;
    fout << "AllcatType="               << prefs->allcatType << endl;
    fout << "ShowAllNotCats="           << prefs->showAllNotCats << endl;
    fout << "DisableKnownClientList="   << prefs->m_bDisableKnownClientList << endl;
    fout << "DisableQueueList="         << prefs->m_bDisableQueueList << endl;
    fout << "UseCreditSystem="          << prefs->m_bCreditSystem << endl;
    fout << "SaveLogToDisk="            << prefs->log2disk << endl;
    fout << "SaveDebugToDisk="          << prefs->debug2disk << endl;
    fout << "EnableScheduler="          << prefs->scheduler << endl;
    fout << "MessagesFromFriendsOnly="  << prefs->msgonlyfriends << endl;
    fout << "MessageFromValidSourcesOnly=" << prefs->msgsecure << endl;
    fout << "ShowInfoOnCatTabs="        << prefs->showCatTabInfos << endl;
    fout << "ResumeNextFromSameCat="    << prefs->resumeSameCat << endl;
    fout << "DontRecreateStatGraphsOnResize=" << prefs->dontRecreateGraphs << endl;
    fout << "VersionCheckLastAutomatic=" << prefs->versioncheckLastAutomatic << endl;
    fout << "FilterLevel="              << prefs->filterlevel << endl;
    fout << "server_list-cols="          << DynPrefs::Get<wxString>("server_list-cols").mb_str(*wxConvCurrent) << endl;
    fout << "download_list-cols="        << DynPrefs::Get<wxString>("download_list-cols").mb_str(*wxConvCurrent) << endl;
    fout << "upload_list-cols="          << DynPrefs::Get<wxString>("upload_list-cols").mb_str(*wxConvCurrent) << endl;
    fout << "queue_list-cols="           << DynPrefs::Get<wxString>("queue_list-cols").mb_str(*wxConvCurrent) << endl;
    fout << "shared_list-cols="          << DynPrefs::Get<wxString>("shared_list-cols").mb_str(*wxConvCurrent) << endl;

    fout << "systray-integration="       << DynPrefs::Get<long>("systray-integration") << endl;
    fout << "accept-license="            << DynPrefs::Get<bool>("accept-license") << endl;
    
/* --- Statistics --- */
    fout << "[Statistics]" << endl;
    for (int i = 0 ; i < 13 ; i++)
    {
        fout << "StatColor" << i << "=" << GetStatsColor(i) << endl;
    }

    fout << "TotalDownloadedBytes="     << prefs->totalDownloadedBytes << endl;
    fout << "TotalUploadedBytes="       << prefs->totalUploadedBytes << endl;

/* --- Razor Prefs --- */
    fout << "[Razor_Preferences]" << endl;
    fout << "NoNeededSources="          << prefs->DropNoNeededSources << endl;
    fout << "SwapNoNeededSources="      << prefs->SwapNoNeededSources << endl;
    fout << "FullQueueSources="         << prefs->DropFullQueueSources << endl;
    fout << "HighQueueRankingSources="  << prefs->DropHighQueueRankingSources << endl;
    fout << "HighQueueRanking="         << prefs->HighQueueRanking << endl;
    fout << "AutoDropTimer="            << prefs->AutoDropTimer << endl;
    fout << "main-x-pos="               << DynPrefs::Get<long>("main-x-pos") << endl;
    fout << "main-y-pos="               << DynPrefs::Get<long>("main-y-pos") << endl;
    fout << "main-x-size="              << DynPrefs::Get<long>("main-x-size") << endl;
    fout << "main-y-size="              << DynPrefs::Get<long>("main-y-size") << endl;
    
    fout.close();
}

bool CPreferences::LoadPreferences(void)
{
    bool error = false;
    bool NoColor = true;    
    wxString buffer2;
    char buffer[200];
    char* internalbuffer;
    //Open File for Test
    wxString full_path(getenv("HOME"), *wxConvCurrent);

    full_path += wxT("/.eMule");

    error = wxFileExists(full_path) == false;

    std::string filename = getenv("HOME");
    filename += "/.xMule/prefs.ini";

    dictionary* ini;
    ini = iniparser_load(const_cast<char *>(filename.c_str()));
    DynPrefs::Add("nickname",               wxString(iniparser_getstring(ini, "xMule:nickname", "http://www.xmule.ws/"), wxConvUTF8));
    // Use CIni anyway to load or to create the values
    DynPrefs::Add("accept-license",         static_cast<bool>(iniparser_getboolean(ini, "xMule:accept-license", false)));
    DynPrefs::Add("nickname",               iniparser_getstring(ini, "xMule:nickname", "http://www.xmule.ws/"));
    internalbuffer = nstrdup(wxString(DynPrefs::Get<wxString>("app-directory") + wxT("Incoming")).mb_str(*wxConvCurrent));
    DynPrefs::Add("incoming-directory",     wxString(iniparser_getstring(ini, "xMule:incoming-directory", internalbuffer), *wxConvCurrent));
    delete[] internalbuffer;
    internalbuffer = nstrdup(wxString(DynPrefs::Get<wxString>("app-directory") + wxT("Temp")).mb_str(*wxConvCurrent));
    DynPrefs::Add("temp-directory",         wxString(iniparser_getstring(ini, "xMule:temp-directory", internalbuffer), *wxConvCurrent));
    delete[] internalbuffer;
    DynPrefs::Add("limits-upload",          static_cast<long>(iniparser_getint(ini, "xMule:limits-upload", UNLIMITED)));
    DynPrefs::Add("limits-download",        static_cast<long>(iniparser_getint(ini, "xMule:limits-download", UNLIMITED)));
    DynPrefs::Add("max-connections",        static_cast<long>(iniparser_getint(ini, "xMule:max-connections", 500)));
    DynPrefs::Add("remove-dead-server",     static_cast<bool>(iniparser_getint(ini, "xMule:remove-dead-server", 2)));
    DynPrefs::Add("tcp-port",               static_cast<long>(iniparser_getint(ini, "xMule:tcp-port", 4662)));
    DynPrefs::Add("udp-port",               static_cast<long>(iniparser_getint(ini, "xMule:udp-port", DynPrefs::Get<long>("tcp-port") + 10)));
    DynPrefs::Add("max-sources",            static_cast<long>(iniparser_getint(ini, "xMule:max-sources", 400)));
    DynPrefs::Add("see-share",              static_cast<long>(iniparser_getint(ini, "xMule:see-share", 2)));
    prefs->m_iToolDelayTime = iniparser_getint(ini, "xMule:ToolTipDelay", 1);
    DynPrefs::Add("graph-delay",            static_cast<long>(iniparser_getint(ini, "xMule:graph-delay", 3)));
    DynPrefs::Add("tree-delay",             static_cast<long>(iniparser_getint(ini, "xMule:tree-delay", 5)));
    DynPrefs::Add("capacities-download",    static_cast<long>(iniparser_getint(ini, "xMule:capacities-download", 96)));
    DynPrefs::Add("capacities-upload",      static_cast<long>(iniparser_getint(ini, "xMule:capacities-upload", 16)));
    DynPrefs::Add("server-connect-retries", static_cast<long>(iniparser_getint(ini, "xMule:server-connect-retries", 1)));
    prefs->splitterbarPosition = iniparser_getint(ini, "xMule:SplitterbarPosition", 75);
    DynPrefs::Add("graph-average",          static_cast<long>(iniparser_getint(ini, "xMule:graph-average", 5)));
    DynPrefs::Add("max-per-5",              static_cast<long>(iniparser_getint(ini, "xMule:max-per-5", GetDefaultMaxConperFive())));
    DynPrefs::Add("reconnect-on-loss",      static_cast<bool>(iniparser_getboolean(ini, "xMule:reconnect-on-loss", true)));
    DynPrefs::Add("priority-system",        static_cast<bool>(iniparser_getboolean(ini, "xMule:priority-system", true)));
    DynPrefs::Add("enable-ich",             static_cast<bool>(iniparser_getboolean(ini, "xMule:enable-ich", true)));
    DynPrefs::Add("autoupdate-server",      static_cast<bool>(iniparser_getboolean(ini, "xMule:autoupdate-server", false)));    
    prefs->updatenotify = static_cast<bool>(iniparser_getboolean(ini, "xMule:UpdateNotify", false));
    DynPrefs::Add("minimize-to-tray",       static_cast<bool>(iniparser_getboolean(ini, "xMule:minimize-to-tray", false)));
    DynPrefs::Add("add-server-server",      static_cast<bool>(iniparser_getboolean(ini, "xMule:add-server-server", true)));
    DynPrefs::Add("add-server-client",      static_cast<bool>(iniparser_getboolean(ini, "xMule:add-server-client", true)));
    DynPrefs::Add("show-splashscreen",      static_cast<bool>(iniparser_getboolean(ini, "xMule:show-splashscreen", true)));
    DynPrefs::Add("bring-to-front",         static_cast<bool>(iniparser_getboolean(ini, "xMule:bring-to-front", true)));
    DynPrefs::Add("download-doubleclick",   static_cast<bool>(iniparser_getboolean(ini, "xMule:download-doubleclick", true)));
    DynPrefs::Add("beep-on-errors",         static_cast<bool>(iniparser_getboolean(ini, "xMule:beep-on-errors", true)));
    DynPrefs::Add("prompt-on-exit",         static_cast<bool>(iniparser_getboolean(ini, "xMule:prompt-on-exit", false)));
    DynPrefs::Add("filter-bad-ip",          static_cast<bool>(iniparser_getboolean(ini, "xMule:filter-bad-ip", true)));
    DynPrefs::Add("autoconnect",            static_cast<bool>(iniparser_getboolean(ini, "xMule:autoconnect", false)));
    DynPrefs::Add("xfer-on-title",          static_cast<bool>(iniparser_getboolean(ini, "xMule:xfer-on-title", false)));
    DynPrefs::Add("online-signature",       static_cast<bool>(iniparser_getboolean(ini, "xMule:online-signature", false)));
    DynPrefs::Add("start-minimized",        static_cast<bool>(iniparser_getboolean(ini, "xMule:start-minimized", false)));
    DynPrefs::Add("use-safe-connect",       static_cast<bool>(iniparser_getboolean(ini, "xMule:use-safe-connect", false)));
    prefs->filterlevel = iniparser_getint(ini, "xMule:FilterLevel", 127);
    DynPrefs::Add("auto-static",            static_cast<bool>(iniparser_getboolean(ini, "xMule:auto-static", false)));
    prefs->autotakeed2klinks = static_cast<bool>(iniparser_getboolean(ini, "xMule:AutoTakeED2KLinks", true));
    DynPrefs::Add("add-paused",             static_cast<bool>(iniparser_getboolean(ini, "xMule:add-paused", false)));
    DynPrefs::Add("progress-bar-style",     static_cast<long>(iniparser_getint(ini, "xMule:progress-bar-style", 0)));
    prefs->useDownloadNotifier = static_cast<bool>(iniparser_getboolean(ini, "xMule:NotifyOnDownload", false));
    prefs->useChatNotifier = static_cast<bool>(iniparser_getboolean(ini, "xMule:NotifyOnChat", false));
    prefs->useLogNotifier = static_cast<bool>(iniparser_getboolean(ini, "xMule:NotifyOnLog", false));
    prefs->useSoundInNotifier = static_cast<bool>(iniparser_getboolean(ini, "xMule:NotifierUseSound", false));
    DynPrefs::Add("smart-lowid-check",      static_cast<bool>(iniparser_getboolean(ini, "xMule:smart-lowid-check", true)));
    prefs->m_bVerbose = static_cast<bool>(iniparser_getboolean(ini, "xMule:Verbose", false));
    DynPrefs::Add("last-chunk-first",       static_cast<bool>(iniparser_getboolean(ini, "xMule:last-chunk-first", false)));
    DynPrefs::Add("realtime-queue",         static_cast<bool>(iniparser_getboolean(ini, "xMule:realtime-queue", false)));
    DynPrefs::Add("add-server-high",        static_cast<bool>(iniparser_getboolean(ini, "xMule:add-server-high", false)));
    DynPrefs::Add("transfer-full-chunks",   static_cast<bool>(iniparser_getboolean(ini, "xMule:transfer-full-chunks", true)));
    DynPrefs::Add("start-next-file",        static_cast<bool>(iniparser_getboolean(ini, "xMule:start-next-file", false)));
    DynPrefs::Add("show-overhead",          static_cast<bool>(iniparser_getboolean(ini, "xMule:show-overhead", false)));
    prefs->moviePreviewBackup = static_cast<bool>(iniparser_getboolean(ini, "xMule:VideoPreviewBackupped", true));
    DynPrefs::Add("file-buffer-size",       static_cast<long>(iniparser_getint(ini, "xMule:file-buffer-size", 16)));
    DynPrefs::Add("upload-queue-size",      static_cast<long>(iniparser_getint(ini, "xMule:upload-queue-size", 50)));
    prefs->versioncheckdays = iniparser_getint(ini, "xMule:Check4NewVersionDelay", 5);
    DynPrefs::Add("auto-priority-share",    static_cast<bool>(iniparser_getboolean(ini, "xMule:auto-priority-share", true)));
    prefs->indicateratings = static_cast<bool>(iniparser_getboolean(ini, "xMule:IndicateRatings", true));
    prefs->allcatType = iniparser_getint(ini, "xMule:AllcatType", 0);
    prefs->showAllNotCats = static_cast<bool>(iniparser_getboolean(ini, "xMule:ShowAllNotCats", false));
    prefs->watchclipboard = static_cast<bool>(iniparser_getboolean(ini, "xMule:WatchClipboard4ED2kFilelinks", false));
    prefs->log2disk = static_cast<bool>(iniparser_getboolean(ini, "xMule:SaveLogToDisk", false));
    prefs->debug2disk = static_cast<bool>(iniparser_getboolean(ini, "xMule:SaveDebugToDisk", false));
    prefs->iMaxLogMessages = iniparser_getint(ini, "xMule:MaxLogMessages", 1000);
    prefs->showCatTabInfos = static_cast<bool>(iniparser_getboolean(ini, "xMule:ShowInfoOnCatTabs", false));
    prefs->resumeSameCat = static_cast<bool>(iniparser_getboolean(ini, "xMule:ResumeNextFromSameCat", false));
    prefs->resumeSameCat = static_cast<bool>(iniparser_getboolean(ini, "xMule:DontRecreateStatGraphsOnResize", false));
    prefs->versioncheckLastAutomatic = iniparser_getint(ini, "xMule:VersionCheckLastAutomatic", 0);
    prefs->m_bDisableKnownClientList = iniparser_getint(ini, "xMule:DisableKnownClientList", false);
    prefs->m_bDisableQueueList = iniparser_getint(ini, "xMule:DisableQueueList", false);
    prefs->m_bCreditSystem = iniparser_getint(ini, "xMule:UseCreditSystem", true);
    prefs->scheduler = static_cast<bool>(iniparser_getboolean(ini, "xMule:EnableScheduler", false));
    prefs->msgonlyfriends = static_cast<bool>(iniparser_getboolean(ini, "xMule:MessagesFromFriendsOnly", false));
    prefs->msgsecure = static_cast<bool>(iniparser_getboolean(ini, "xMule:MessageFromValidSourcesOnly", true));
    prefs->maxmsgsessions = iniparser_getint(ini, "xMule:MaxMessageSessions", 50);
    sprintf(prefs->VideoPlayer, "%s", iniparser_getstring(ini, "xMule:VideoPlayer", ""));
    DynPrefs::Add("server_list-cols",       wxString(iniparser_getstring(ini, "xMule:server_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("download_list-cols",     wxString(iniparser_getstring(ini, "xMule:download_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("upload_list-cols",       wxString(iniparser_getstring(ini, "xMule:upload_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("queue_list-cols",        wxString(iniparser_getstring(ini, "xMule:queue_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("friend_list-cols",       wxString(iniparser_getstring(ini, "xMule:friend_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("shared_list-cols",       wxString(iniparser_getstring(ini, "xMule:shared_list-cols", ""), *wxConvCurrent));
    DynPrefs::Add("systray-integration",    static_cast<long>(iniparser_getint(ini, "xMule:systray-integration", 0)));
    
    for (int i = 0 ; i < 13 ; i++)
    {
         std::ostringstream sout;
         sout << "StatColor" << i;
         sprintf(buffer, "%s", iniparser_getstring(ini, const_cast<char *>(sout.str().c_str()), "0"));
         if ((prefs->statcolors[i] = atoll(buffer)))
         {
             NoColor = false;
         }
    }
    
    if (NoColor)
    {
         for (int i = 0 ; i < 13 ; i++)
         {
              ResetStatsColor(i);
         }
    }
    
    prefs->totalDownloadedBytes = iniparser_getint(ini, "Statistics:TotalDownloadedBytes", 0);
    prefs->totalUploadedBytes = iniparser_getint(ini, "Statistics:TotalUploadedBytes", 0);
    prefs->DropNoNeededSources = static_cast<bool>(iniparser_getboolean(ini, "Razor_Preferences:NoNeededSources", false));
    prefs->SwapNoNeededSources = static_cast<bool>(iniparser_getboolean(ini, "Razor_Preferences:SwapNoNeededSources", false));
    prefs->DropFullQueueSources = static_cast<bool>(iniparser_getboolean(ini, "Razor_Preferences:FullQueueSources", false));
    prefs->DropHighQueueRankingSources = static_cast<bool>(iniparser_getboolean(ini, "Razor_Preferences:HighQueueRankingSources", false));
    prefs->HighQueueRanking = iniparser_getint(ini, "Razor_Preferences:HighQueueRanking", 1200);
    prefs->AutoDropTimer = iniparser_getint(ini, "Razor_Preferences:AutoDropTimer", 240);
    DynPrefs::Add("main-x-pos",             static_cast<long>(iniparser_getint(ini, "main-x-pos", 0)));
    DynPrefs::Add("main-y-pos",             static_cast<long>(iniparser_getint(ini, "main-y-pos", 0)));
    DynPrefs::Add("main-x-size",            static_cast<long>(iniparser_getint(ini, "main-x-size", 800)));
    DynPrefs::Add("main-y-size",            static_cast<long>(iniparser_getint(ini, "main-y-size", 600)));

    return error;
}

bool CPreferences::Load_Adresses(void)
{
    //serverlist adresses
    bool error = false;
    std::ostringstream full_path;
    full_path << DynPrefs::Get<wxString>("app-directory").mb_str(*wxConvCurrent) << "adresses.dat";
    FILE *sdirfile = fopen(full_path.str().c_str(), "r");

    if (sdirfile)
    {
        char buffer[MAX_PATH];

        while (!feof(sdirfile))
        {
            memset(buffer, 0, sizeof(buffer));
            fgets(buffer, sizeof(buffer) - 1, sdirfile);
            char *ptr = strchr(buffer, '\n');

            if (ptr)
            {
                *ptr = 0;
            }

            if (strlen(buffer) > 1)
            {
                addresses_list.push_back(wxString(buffer, *wxConvCurrent));
            }
        }

        fclose(sdirfile);
    }
    else
    {
        error = true;
    }

    return error;
}

bool CPreferences::Save_Adresses(void)
{
    bool error = false;
    ; // I am to lazy atm :)
    return error;
}

bool CPreferences::Load_Shared_Dirs(void)
{
    bool error = false;
    std::ostringstream full_path;
    full_path << DynPrefs::Get<wxString>("app-directory").mb_str(*wxConvCurrent) << "shareddir.dat";
    FILE *sdirfile = fopen(full_path.str().c_str(), "r");
    if (sdirfile)
    {
        char buffer[MAX_PATH];
        while (!feof(sdirfile))
        {
            memset(buffer, 0, sizeof(buffer));
            fgets(buffer, sizeof(buffer) - 1, sdirfile);
            char *ptr = strchr(buffer, '\n');
            if (ptr) *ptr = 0;
            if (strlen(buffer) > 1)
            shareddir_list.Add(wxString(buffer, *wxConvCurrent));
        }
        fclose(sdirfile);
    }
    else
    {
        error = true;
    }
    return error;
}

bool CPreferences::Save_Shared_Dirs(void)
{
    bool error = false;
    std::ostringstream full_path;
    full_path << DynPrefs::Get<wxString>("app-directory").mb_str(*wxConvCurrent) << "shareddir.dat";
    FILE *sdirfile = fopen(full_path.str().c_str(), "w");
    if (sdirfile)
    {
        for (int i = 0 ; i < shareddir_list.GetCount() ; i++)
        {
            fprintf(sdirfile, "%s\n", (const char *)(shareddir_list[i].mb_str(*wxConvCurrent)));
        }
        fclose(sdirfile);
    }
    else
    {
        error = true;
    }
    return error;
}


bool CPreferences::SaveCats(void)
{
    bool error = false;
    ofstream fout;
    wxString buffer;
    wxString catinif(DynPrefs::Get<wxString>("app-directory") + wxT("Category.dat"));

    if (GetCatCount() > 0)
    {
        fout.open(catinif.mb_str(*wxConvCurrent));

        if (fout.good() == true)
        {
            fout << catMap.GetCount() - 1 << " ";

            for (int ix = 1; ix < catMap.GetCount(); ++ix)
            { 
                fout << catMap.GetAt(ix)->title.mb_str(*wxConvCurrent) << endl;
                fout << catMap.GetAt(ix)->incomingpath.mb_str(*wxConvCurrent) << endl;

                if (catMap.GetAt(ix)->comment != wxT(""))
                {
                    fout << catMap.GetAt(ix)->comment.mb_str(*wxConvCurrent) << endl;
                }
                else
                {
                    fout << "-1" << endl;
                }

                fout << "0x" << std::hex << catMap.GetAt(ix)->color << endl;
                fout << std::setprecision(3) << std::fixed << catMap.GetAt(ix)->prio + 1 << endl;
            }

            fout.close();
        } 
        else
        {
            error = true;
        }
    }
    return error;
}
#include <iostream>
using std::cout;
bool CPreferences::LoadCats(void)
{
    bool error = false;
    int max;

    // default cat
    Category_Struct* newcat = new Category_Struct;
    newcat->prio = 0;
    newcat->color = 0;
    AddCat(newcat);
    // Load cats
    wxString catinif(DynPrefs::Get<wxString>("app-directory") + wxT("Category.dat"));

    ifstream fin(catinif.mb_str(*wxConvCurrent));
    
    if (fin.good() == true)
    {
        fin >> max;

        std::string temp;

        for (int i = 1; i <= max; ++i)
        {
            Category_Struct* newcat = new Category_Struct;

            fin >> temp;
            newcat->title = wxString(temp.c_str(), *wxConvCurrent);

            fin >> temp;
            newcat->incomingpath = wxString(temp.c_str(), *wxConvCurrent);

            fin >> temp;

            if (temp == "-1")
            {
                newcat->comment = wxT("");
            }
            else
            {
                newcat->comment = wxString(temp.c_str(), *wxConvCurrent);
            }

            fin >> std::hex >> newcat->color;
            fin >> std::dec >> newcat->prio;

            AddCat(newcat);

            if (!wxFileName::DirExists(newcat->incomingpath)) 
            {
                wxMkdir(newcat->incomingpath, 0777);
            }
        }

        fin.close();
    } 
    else
    {
       error = true;
    }
    return error;
}

void CPreferences::RemoveCat(int index)
{
    if (index >= 0 &&index < catMap.GetCount())
    {
        Category_Struct *delcat;
        delcat = catMap.GetAt(index);
        catMap.RemoveAt(index);
        delete delcat;
    }
}

